package ru.yandex.travel.orders.workflows.orderitem.bus.ticketrefund.handlers;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.bus.service.BusesService;
import ru.yandex.travel.bus.service.BusesServiceRetryableException;
import ru.yandex.travel.buses.backend.proto.worker.TRefundRequest;
import ru.yandex.travel.buses.backend.proto.worker.TRefundResponse;
import ru.yandex.travel.buses.backend.proto.worker.TRequestHeader;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.entities.BusTicketRefund;
import ru.yandex.travel.orders.services.buses.BusesServiceProvider;
import ru.yandex.travel.orders.workflow.orderitem.bus.ticketrefund.proto.EBusTicketRefundState;
import ru.yandex.travel.orders.workflow.orderitem.bus.ticketrefund.proto.TRefundCommit;
import ru.yandex.travel.orders.workflow.orderitem.bus.ticketrefund.proto.TTicketRefundFailed;
import ru.yandex.travel.orders.workflow.orderitem.bus.ticketrefund.proto.TTicketRefundSuccess;
import ru.yandex.travel.orders.workflows.orderitem.bus.BusProperties;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.exceptions.RetryableException;

@Slf4j
@RequiredArgsConstructor
public class RefundingStateHandler
        extends AnnotatedStatefulWorkflowEventHandler<EBusTicketRefundState, BusTicketRefund> {
    private final BusesServiceProvider busesServiceProvider;
    private final BusProperties busProperties;

    @HandleEvent
    public void handleStartRefund(TRefundCommit message,
                                  StateContext<EBusTicketRefundState, BusTicketRefund> ctx) {
        BusTicketRefund busTicketRefund = ctx.getWorkflowEntity();
        String ticketId = busTicketRefund.getPayload().getTicketId();
        TRefundRequest refundRequest = TRefundRequest.newBuilder()
                .setHeader(TRequestHeader.newBuilder())
                .setSupplierId(busTicketRefund.getPayload().getSupplierId())
                .setOrderId(busTicketRefund.getPayload().getOrderId())
                .setTicketId(ticketId)
                .build();
        TRefundResponse refundResponse;
        BusesService busesService = busesServiceProvider.getBusesServiceByOrderItem(busTicketRefund.getOrderItem());
        try {
            refundResponse = busesService.refund(refundRequest);
        } catch (Exception e) {
            if (e instanceof BusesServiceRetryableException &&
                    ctx.getAttempt() < busProperties.getRefundCommitMaxTries()) {
                throw new RetryableException(e, busProperties.getRefundCommitRetryDelay());
            }
            log.warn("Refund error for ticket {}: {}", ticketId, e);
            ctx.setState(EBusTicketRefundState.RS_FAILED);
            ctx.scheduleExternalEvent(busTicketRefund.getOrderItemWorkflowId(), TTicketRefundFailed.newBuilder()
                    .setRefundId(busTicketRefund.getOrderRefund().getId().toString())
                    .setTicketRefundId(busTicketRefund.getId().toString())
                    .setTicketId(ticketId).build());
            return;
        }
        Preconditions.checkState(refundResponse.getRefund().hasPrice(), "Price expected");
        busTicketRefund.getPayload().setRefundAmount(ProtoUtils.fromTPrice(refundResponse.getRefund().getPrice()));

        ctx.setState(EBusTicketRefundState.RS_REFUNDED);
        ctx.scheduleExternalEvent(busTicketRefund.getOrderItemWorkflowId(), TTicketRefundSuccess.newBuilder()
                .setRefundId(busTicketRefund.getOrderRefund().getId().toString())
                .setTicketRefundId(busTicketRefund.getId().toString())
                .setTicketId(busTicketRefund.getPayload().getTicketId())
                .setPrice(refundResponse.getRefund().getPrice())
                .build());
    }
}
