package ru.yandex.travel.orders.workflows.orderitem.dolphin;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDate;
import java.util.List;
import java.util.stream.Collectors;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.ErrorException;
import ru.yandex.travel.commons.proto.TError;
import ru.yandex.travel.hotels.common.partners.base.CallContext;
import ru.yandex.travel.hotels.common.partners.base.exceptions.PartnerException;
import ru.yandex.travel.hotels.common.partners.base.exceptions.RetryableHttpException;
import ru.yandex.travel.hotels.common.partners.base.exceptions.RetryableIOException;
import ru.yandex.travel.hotels.common.partners.dolphin.DolphinClient;
import ru.yandex.travel.hotels.common.partners.dolphin.model.AnnulateResult;
import ru.yandex.travel.hotels.common.partners.dolphin.model.CreateOrderRequest;
import ru.yandex.travel.hotels.common.partners.dolphin.model.Order;
import ru.yandex.travel.hotels.common.partners.dolphin.model.OrderState;
import ru.yandex.travel.hotels.common.partners.dolphin.model.OrdersInfoRequest;
import ru.yandex.travel.hotels.common.partners.dolphin.model.QueryPeriod;
import ru.yandex.travel.orders.entities.DolphinOrderItem;
import ru.yandex.travel.orders.entities.PriceCheckOutcome;
import ru.yandex.travel.orders.management.StarTrekConfigurationProperties;
import ru.yandex.travel.orders.management.StarTrekService;
import ru.yandex.travel.orders.repository.DolphinOrderItemRepository;
import ru.yandex.travel.orders.services.RefundCalculationService;
import ru.yandex.travel.orders.workflow.hotels.dolphin.proto.EDolphinItemState;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.exceptions.RetryableException;

@Component
@RequiredArgsConstructor
@Slf4j
public class DolphinService {
    private final DolphinClient dolphinClient;
    @Getter
    private final DolphinProperties properties;
    private final DolphinOrderItemRepository repository;
    private final RefundCalculationService refundCalculationService;
    private final StarTrekConfigurationProperties starTrekConfigurationProperties;
    private final StarTrekService starTrekService;

    public Order findExistingOrder(CreateOrderRequest requestToCreate, CallContext callContext) throws DuplicateDolphinOrderException {
        OrdersInfoRequest searchRequest = OrdersInfoRequest.builder()
                .createPeriod(new QueryPeriod(LocalDate.now().minusDays(1), LocalDate.now().plusDays(1)))
                .state(OrderState.OK)
                .state(OrderState.IN_WORK)
                .state(OrderState.WAIT_LIST)
                .state(OrderState.UNDEFINED)
                .build();
        log.info("Looking up for existing order for request {}", requestToCreate);
        List<Order> duplicates;
        try {
            duplicates = dolphinClient.withCallContext(callContext).getOrdersSync(searchRequest).stream()
                    .filter(o -> DolphinServiceHelper.createRequestsMatch(o.getRequest(), requestToCreate))
                    .collect(Collectors.toList());
        } catch (PartnerException exception) {
            return handleDolphinException(exception);
        }
        if (duplicates.isEmpty()) {
            log.debug("No existing orders found, will create");
            return null;
        }
        if (duplicates.size() > 1) {
            var msg = String.format("%s existing dolphin orders found", duplicates.size());
            log.error(msg);
            throw new RuntimeException("Unexpected number of duplicate orders");
        } else {
            String code = duplicates.get(0).getCode();
            log.info("Found an already existing dolphin order with code {}", code);
            var existing = repository.getByDolphinOrderCode(code);
            if (existing != null) {
                var msg = String.format("Duplicate dolphin order with code %s is associated with another order %s",
                        code, existing.getOrder().getId());
                log.error(msg);
                throw new DuplicateDolphinOrderException(existing.getOrder().getPrettyId(), code);
            } else {
                log.info("Dolphin order with code {} is not associated with any our order. Will attach to a new " +
                                "one",
                        code);
                return duplicates.get(0);
            }
        }
    }

    public PriceCheckOutcome checkPriceMismatchOrThrow(BigDecimal actualAmount, BigDecimal expectedAmount,
                                                       StateContext<EDolphinItemState, DolphinOrderItem> stateContext) {
        BigDecimal difference = actualAmount.divide(expectedAmount, 2, RoundingMode.HALF_UP).subtract(BigDecimal.ONE);
        BigDecimal maxDiff = BigDecimal.valueOf(properties.getMaxPriceMismatchThreshold());
        if (difference.compareTo(BigDecimal.ZERO) != 0) {
            log.warn("Price mismatch on created order: expected {}, got {}",
                    expectedAmount.setScale(2, RoundingMode.HALF_UP).toString(),
                    actualAmount.setScale(2, RoundingMode.HALF_UP).toString());
            boolean exceedsLimit = difference.compareTo(maxDiff) > 0;
            boolean isOrderRefundable =
                    refundCalculationService.isCurrentlyFullyRefundable(stateContext.getWorkflowEntity().getOrder());
            PriceCheckOutcome outcome;
            if (!exceedsLimit) {
                log.warn("Price mismatch does not exceed limit: will confirm the order");
                outcome = PriceCheckOutcome.CONFIRMED;
            } else {
                if (isOrderRefundable) {
                    log.warn("Price mismatch exceeds the limit, order is refundable: will cancel the order");
                    outcome = PriceCheckOutcome.CANCELLED;
                } else {
                    log.warn("Price mismatch exceeds the limit, order is non refundable: crashing");
                    outcome = PriceCheckOutcome.CRASHED;
                }
            }

            if (outcome == PriceCheckOutcome.CRASHED) {
                throw new ErrorException(TError.newBuilder()
                        .setMessage("Расхождение цены превышает допустимое")
                        .putAttribute("Ожидаемая цена", expectedAmount.setScale(2, RoundingMode.HALF_UP).toString())
                        .putAttribute("Реальная цена", actualAmount.setScale(2, RoundingMode.HALF_DOWN).toString())
                        .putAttribute("Заказ возвратный?", "нет")
                        .setCode(EErrorCode.EC_FAILED_PRECONDITION)
                        .build());
            } else {
                starTrekService.createIssueForHotelOrderPriceMismatch(stateContext.getWorkflowEntity().getOrder(),
                        actualAmount, expectedAmount, exceedsLimit, isOrderRefundable, outcome, stateContext);
            }
            return outcome;
        } else {
            return PriceCheckOutcome.CONFIRMED;
        }
    }

    public Order createAndGet(CreateOrderRequest request, CallContext callContext) {
        try {
            return dolphinClient.withCallContext(callContext).createOrderSync(request);
        } catch (PartnerException exception) {
            return handleDolphinException(exception);
        }
    }

    public AnnulateResult annulateOrder(String code, CallContext callContext) {
        try {
            return dolphinClient.withCallContext(callContext).annulateOrderSync(code);
        } catch (PartnerException exception) {
            return handleDolphinException(exception);
        }
    }

    public Order get(String code, CallContext callContext) {
        try {
            return dolphinClient.withCallContext(callContext).getOrderSync(code);
        } catch (PartnerException exception) {
            return handleDolphinException(exception);
        }
    }

    private <T> T handleDolphinException(PartnerException exception) {
        Throwable cause = exception.getCause();
        if (cause == null) {
            cause = exception;
        }
        if (cause instanceof RetryableHttpException) {
            throw new RetryableException(cause);
        } else if (cause instanceof RetryableIOException) {
            throw new RetryableException(cause);
        } else {
            throw new RuntimeException(cause);
        }
    }
}
