package ru.yandex.travel.orders.workflows.orderitem.dolphin;

import com.google.common.collect.ImmutableMap;
import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.hotels.common.partners.dolphin.DefaultDolphinClient;
import ru.yandex.travel.hotels.common.partners.dolphin.DolphinClient;
import ru.yandex.travel.hotels.common.partners.dolphin.WrappedDolphinClient;
import ru.yandex.travel.orders.entities.DolphinOrderItem;
import ru.yandex.travel.orders.services.finances.providers.DolphinFinancialDataProviderProperties;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.services.partners.BillingPartnerService;
import ru.yandex.travel.orders.workflow.hotels.dolphin.proto.EDolphinItemState;
import ru.yandex.travel.orders.workflows.orderitem.dolphin.handlers.CancelledStateHandler;
import ru.yandex.travel.orders.workflows.orderitem.dolphin.handlers.CancellingStateHandler;
import ru.yandex.travel.orders.workflows.orderitem.dolphin.handlers.ConfirmedStateHandler;
import ru.yandex.travel.orders.workflows.orderitem.dolphin.handlers.ConfirmingStateHandler;
import ru.yandex.travel.orders.workflows.orderitem.dolphin.handlers.NewStateHandler;
import ru.yandex.travel.orders.workflows.orderitem.dolphin.handlers.PollingStateHandler;
import ru.yandex.travel.orders.workflows.orderitem.dolphin.handlers.RefundingStateHandler;
import ru.yandex.travel.orders.workflows.orderitem.dolphin.handlers.ReservedStateHandler;
import ru.yandex.travel.workflow.StatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.WorkflowEventHandler;
import ru.yandex.travel.workflow.base.ProxyStateMapWorkflowEventHandler;
import ru.yandex.travel.workflow.repository.EntityStateTransitionRepository;

@Configuration
@EnableConfigurationProperties(DolphinProperties.class)
@RequiredArgsConstructor
public class DolphinWorkflowConfiguration {
    private final EntityStateTransitionRepository entityStateTransitionRepository;
    private final DolphinProperties dolphinProperties;
    private final ManualTicketGenerator ticketGenerator;
    private final Meters meters;
    private final DolphinFiscalItemCreator fiscalItemCreator;

    @Bean
    public WorkflowEventHandler<DolphinOrderItem> dolphinOrderItemWorkflowEventHandler(
            DolphinService dolphinService,
            BillingPartnerService billingPartnerService
    ) {
        ImmutableMap.Builder<EDolphinItemState, StatefulWorkflowEventHandler<EDolphinItemState, DolphinOrderItem>> builder = ImmutableMap.builder();
        builder
                .put(EDolphinItemState.IS_NEW, new NewStateHandler(billingPartnerService, dolphinProperties, meters,
                        fiscalItemCreator))
                .put(EDolphinItemState.IS_RESERVED, new ReservedStateHandler(dolphinService, meters, ticketGenerator))
                .put(EDolphinItemState.IS_CONFIRMING, new ConfirmingStateHandler(dolphinService, meters, ticketGenerator))
                .put(EDolphinItemState.IS_CONFIRMED, new ConfirmedStateHandler(dolphinService, meters, ticketGenerator))
                .put(EDolphinItemState.IS_REFUNDING, new RefundingStateHandler(dolphinService, meters, ticketGenerator))
                .put(EDolphinItemState.IS_CANCELLING, new CancellingStateHandler(dolphinService, meters, ticketGenerator))
                .put(EDolphinItemState.IS_CANCELLED, new CancelledStateHandler())
                .put(EDolphinItemState.IS_POLLING_FOR_STATUS, new PollingStateHandler(dolphinService, meters, ticketGenerator));
        return new ProxyStateMapWorkflowEventHandler<>(builder.build(), entityStateTransitionRepository);
    }

    @Bean
    public DolphinClient dolphinClient(@Qualifier("dolphinAhcClientWrapper") AsyncHttpClientWrapper dolphinClientWrapper,
                                       Retry retryHelper) {
        DefaultDolphinClient defaultDolphinClient = new DefaultDolphinClient(dolphinClientWrapper,
                dolphinProperties.getClient(), retryHelper);
        if (dolphinProperties.isEnableTestNames()) {
            return new WrappedDolphinClient(defaultDolphinClient);
        } else {
            return defaultDolphinClient;
        }
    }

    @Bean
    public DolphinFinancialDataProviderProperties dolphinFinancialDataProviderProperties(DolphinProperties dolphinProperties) {
        return dolphinProperties.getFinancialData();
    }
}
