package ru.yandex.travel.orders.workflows.orderitem.dolphin;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import com.google.common.io.Resources;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;

import ru.yandex.travel.hotels.common.orders.OrderDetails;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.management.ManualServiceProcessingTicketData;
import ru.yandex.travel.orders.management.StarTrekConfigurationProperties;
import ru.yandex.travel.orders.management.StarTrekService;
import ru.yandex.travel.orders.repository.TicketRepository;
import ru.yandex.travel.orders.services.RefundCalculationService;
import ru.yandex.travel.orders.workflow.order.proto.TCommentIssue;
import ru.yandex.travel.workflow.MessagingContext;

@Service
@RequiredArgsConstructor
public class ManualTicketGenerator {
    public static final String TEMPLATE_PATH = "support/dolphin";

    private final StarTrekService starTrekService;
    private final StarTrekConfigurationProperties properties;
    private final RefundCalculationService refundCalculationService;
    private final TicketRepository ticketRepository;

    private String readTemplate(String templateName) {
        String resourcePath = String.format("%s/%s.template", TEMPLATE_PATH, templateName);
        try {
            return Resources.toString(Resources.getResource(resourcePath), StandardCharsets.UTF_8);
        } catch (IOException e) {
            return "Невозможно отобразить инструкцию. Передайте тикет разработчику";
        }
    }

    public UUID generateWaitListTicket(Order order, String code, OrderDetails orderDetails, MessagingContext context) {
        String instructionsTemplate = readTemplate("wait_list");
        var instructions = instructionsTemplate.replaceAll("%ORDER_CODE%", code);

        var data = new ManualServiceProcessingTicketData(
                order.getPrettyId(), "Дельфин", "Попадание в wait-list на стороне партнера",
                refundCalculationService.isCurrentlyFullyRefundable(order), true, true, instructions, new HashMap<>());
        if (orderDetails.getAddressByPartner() != null) {
            data.getExtraDetails().put("Адрес у партнера", orderDetails.getAddressByPartner());
        }
        if (orderDetails.getAddressByYandex() != null) {
            data.getExtraDetails().put("Адрес у нас", orderDetails.getAddressByYandex());
        }
        return starTrekService.createIssueForHotelManualProcessing(order, data, context);
    }

    public UUID generateDuplicateOrderTicket(Order order, String otherOrderPrettyId, String otherOrderCode,
                                             MessagingContext context) {
        String instructionsTemplate = readTemplate("double_order");
        var instructions = instructionsTemplate.replaceAll("%DUPLICATE_ORDER_ID%", otherOrderPrettyId);

        var data = new ManualServiceProcessingTicketData(
                order.getPrettyId(), "Дельфин", "Обнаружен заказ-дубликат на нашей стороне",
                refundCalculationService.isCurrentlyFullyRefundable(order), true, false, instructions,
                Map.of("Id заказа-дубля", otherOrderPrettyId,
                        "Код дубля у партнера", otherOrderCode));
        return starTrekService.createIssueForHotelManualProcessing(order, data, context);
    }

    public void updateWaitlistTicket(UUID ticketId, ManualOutcome outcome, MessagingContext context) {
        var ticket = ticketRepository.getOne(ticketId);
        String comment = null;
        switch (outcome) {
            case CONFIRMED:
                comment = "Заказ восстановлен, бронирование подтверждено успешно";
                break;
            case ANNULLED:
                comment = "Заказ восстановлен, бронирование АННУЛИРОВАНО";
                break;
            case CANCELLED_WITH_NO_REMOTE:
                comment = "Заказ отменен, бронирование не было совершено";
                break;
            case CANCELLED_WITH_ANNULMENT:
                comment = "Заказ отменен, бронирование АННУЛИРОВАНО";
                break;
            case STILL_BROKEN:
                comment = "Не удалось восстановить заказ, он все еще не подтвержден :(";
                break;
        }
        context.scheduleExternalEvent(ticket.getWorkflow().getId(),
                TCommentIssue.newBuilder().setText(comment).build());
    }

    public enum ManualOutcome {
        STILL_BROKEN,
        CONFIRMED,
        ANNULLED,
        CANCELLED_WITH_NO_REMOTE,
        CANCELLED_WITH_ANNULMENT
    }
}
