package ru.yandex.travel.orders.workflows.orderitem.dolphin.handlers;

import java.math.BigDecimal;
import java.time.Instant;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.hotels.common.orders.CancellationDetails;
import ru.yandex.travel.hotels.common.orders.ConfirmationInfo;
import ru.yandex.travel.hotels.common.orders.DolphinHotelItinerary;
import ru.yandex.travel.hotels.common.orders.Guest;
import ru.yandex.travel.hotels.common.partners.base.CallContext;
import ru.yandex.travel.hotels.common.partners.dolphin.model.Order;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.THotelTestContext;
import ru.yandex.travel.orders.entities.DolphinOrderItem;
import ru.yandex.travel.orders.entities.PriceCheckOutcome;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.workflow.hotels.dolphin.proto.EDolphinItemState;
import ru.yandex.travel.orders.workflow.hotels.dolphin.proto.TCancellationCommit;
import ru.yandex.travel.orders.workflow.order.proto.TServiceConfirmed;
import ru.yandex.travel.orders.workflows.orderitem.dolphin.DolphinService;
import ru.yandex.travel.orders.workflows.orderitem.dolphin.ManualTicketGenerator;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;

@RequiredArgsConstructor
@Slf4j
public abstract class BaseDolphinHandler extends AnnotatedStatefulWorkflowEventHandler<EDolphinItemState,
        DolphinOrderItem> {
    protected final DolphinService dolphinService;
    protected final Meters meters;
    protected final ManualTicketGenerator ticketGenerator;

    protected CallContext getCallContext(DolphinOrderItem orderItem, CallContext.CallPhase phase) {
        THotelTestContext testContext = null;
        if (orderItem.getTestContext() != null && orderItem.getTestContext() instanceof THotelTestContext) {
            testContext = (THotelTestContext) orderItem.getTestContext();
        }
        return new CallContext(phase, testContext, null, null, orderItem.getHotelItinerary(), null);
    }


    protected void handleSuccessfulOrderCreation(Order order, DolphinHotelItinerary itinerary,
                                                 StateContext<EDolphinItemState, DolphinOrderItem> context) {
        BigDecimal commission = BigDecimal.valueOf(order.getCost().getFee());
        BigDecimal brutto = BigDecimal.valueOf(order.getCost().getBrutto());
        BigDecimal total = brutto.add(commission);
        BigDecimal expectedTotal = itinerary.getFiscalPrice().getNumberStripped();
        PriceCheckOutcome outcome = dolphinService.checkPriceMismatchOrThrow(total, expectedTotal, context);
        itinerary.setActualPrice(Money.of(total, itinerary.getFiscalPrice().getCurrency()));
        ManualTicketGenerator.ManualOutcome manualOutcome;
        switch (outcome) {
            case CANCELLED:
                context.setState(EDolphinItemState.IS_CANCELLING);
                itinerary.setOrderCancellationDetails(CancellationDetails.create(CancellationDetails.Reason.PRICE_CHANGED));
                meters.incrementCancellationCounter(EPartnerId.PI_DOLPHIN, CancellationDetails.Reason.PRICE_CHANGED);
                context.scheduleEvent(TCancellationCommit.newBuilder().build());
                manualOutcome = ManualTicketGenerator.ManualOutcome.CANCELLED_WITH_ANNULMENT;
                break;
            case CONFIRMED:
                checkAndUpdateGuests(order.getRequest().getGuests(), itinerary.getGuests());
                var confirmationInfo = new ConfirmationInfo();
                confirmationInfo.setPartnerConfirmationId(order.getCode());
                itinerary.setConfirmation(confirmationInfo);
                context.getWorkflowEntity().setConfirmedAt(Instant.now());
                context.setState(EDolphinItemState.IS_CONFIRMED);
                context.scheduleExternalEvent(context.getWorkflowEntity().getOrderWorkflowId(),
                        TServiceConfirmed.newBuilder().setServiceId(context.getWorkflowEntity().getId().toString()).build());
                manualOutcome = ManualTicketGenerator.ManualOutcome.CONFIRMED;
                meters.incrementConfirmationCounter(EPartnerId.PI_DOLPHIN);
                log.info("confirmation info received, itinerary is CONFIRMED");
                break;
            default:
                throw new RuntimeException("Unreachable state reached");
        }
        if (itinerary.getManualTicketId() != null) {
            ticketGenerator.updateWaitlistTicket(itinerary.getManualTicketId(),
                    manualOutcome, context);
        }
    }

    /**
     * Update guests in db based on the data received on order confirmation from dolphin
     *
     * @implNote as we do not send guests without filled names to dolphin, we expect filled guests list to match
     * the list from dolphin
     */
    protected void checkAndUpdateGuests(
            List<ru.yandex.travel.hotels.common.partners.dolphin.model.Guest> confirmedGuests,
            List<Guest> itineraryGuests) {
        Preconditions.checkArgument(
                confirmedGuests.size() <= itineraryGuests.size(),
                String.format("Unexpected amount of guest in order confirmation. %s; while expected no more than %s",
                        confirmedGuests.size(),
                        itineraryGuests.size()));

        var sortedItineraryGuests = itineraryGuests.stream().sorted((guest1, guest2) -> {
            if (guest1.hasFilledName() && !guest2.hasFilledName()) {
                return -1;
            }
            if (!guest1.hasFilledName() && guest2.hasFilledName()) {
                return 1;
            }
            return 0;
        }).collect(Collectors.toList());

        for (int i = 0; i < confirmedGuests.size(); i++) {
            var confirmedGuest = confirmedGuests.get(i);
            var itineraryGuest = sortedItineraryGuests.get(i);
            if (!Objects.equals(confirmedGuest.getCyrillic().getFirstName(), itineraryGuest.getFirstName()) ||
                    !Objects.equals(confirmedGuest.getCyrillic().getLastName(), itineraryGuest.getLastName())) {
                log.warn("Confirmed guest info ({}) does not match the one in itinerary ({}), will update",
                        confirmedGuest, itineraryGuest);
                itineraryGuest.setLastName(confirmedGuest.getCyrillic().getLastName());
                itineraryGuest.setFirstName(confirmedGuest.getCyrillic().getFirstName());
            }
        }
    }
}
