package ru.yandex.travel.orders.workflows.orderitem.dolphin.handlers;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.common.partners.base.CallContext;
import ru.yandex.travel.hotels.common.partners.dolphin.model.OrderState;
import ru.yandex.travel.orders.entities.DolphinOrderItem;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.workflow.hotels.dolphin.proto.EDolphinItemState;
import ru.yandex.travel.orders.workflow.hotels.dolphin.proto.TCancellationCommit;
import ru.yandex.travel.orders.workflow.hotels.dolphin.proto.TReservationExpired;
import ru.yandex.travel.orders.workflow.hotels.proto.TCancellationStart;
import ru.yandex.travel.orders.workflow.hotels.proto.TConfirmationStart;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.orders.workflows.orderitem.dolphin.DolphinService;
import ru.yandex.travel.orders.workflows.orderitem.dolphin.ManualTicketGenerator;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;

@Slf4j
@IgnoreEvents(types = {TCancellationStart.class, TConfirmationStart.class, TReservationExpired.class})
public class CancellingStateHandler extends BaseDolphinHandler {

    public CancellingStateHandler(DolphinService dolphinService, Meters meters, ManualTicketGenerator ticketGenerator) {
        super(dolphinService, meters, ticketGenerator);
    }

    @HandleEvent
    public void handleCancellationCommit(TCancellationCommit message,
                                         StateContext<EDolphinItemState, DolphinOrderItem> context) {
        var itinerary = context.getWorkflowEntity().getItinerary();
        Preconditions.checkNotNull(context.getWorkflowEntity().getDolphinOrderCode(), "Partner order code is not set");
        dolphinService.annulateOrder(context.getWorkflowEntity().getDolphinOrderCode(),
                 getCallContext(context.getWorkflowEntity(), CallContext.CallPhase.ORDER_CANCELLATION));
        var annulatedOrder = dolphinService.get(context.getWorkflowEntity().getDolphinOrderCode(),
                getCallContext(context.getWorkflowEntity(), CallContext.CallPhase.ORDER_CANCELLATION));
        if (annulatedOrder.getState() != OrderState.ANNULATED) {
            var msg = String.format("Itinerary '%s': Unexpected partner state: expected 'ANNULATED', got '%s'",
                    itinerary.getServiceId(), annulatedOrder.getState());
            log.error(msg);
            throw new RuntimeException(msg);
        }
        log.info("partner order is ANNULLED, itinerary is CANCELLED");
        context.setState(EDolphinItemState.IS_CANCELLED);
        context.scheduleExternalEvent(context.getWorkflowEntity().getOrderWorkflowId(),
                TServiceCancelled.newBuilder().setServiceId(context.getWorkflowEntity().getId().toString()).build());
    }
}

