package ru.yandex.travel.orders.workflows.orderitem.dolphin.handlers;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.common.partners.base.CallContext;
import ru.yandex.travel.hotels.common.partners.dolphin.model.OrderState;
import ru.yandex.travel.orders.entities.DolphinOrderItem;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.workflow.hotels.dolphin.proto.EDolphinItemState;
import ru.yandex.travel.orders.workflow.hotels.dolphin.proto.TRefresh;
import ru.yandex.travel.orders.workflow.hotels.dolphin.proto.TRefundCommit;
import ru.yandex.travel.orders.workflow.hotels.proto.TRefundStart;
import ru.yandex.travel.orders.workflow.order.proto.TMoneyAcquired;
import ru.yandex.travel.orders.workflows.orderitem.dolphin.DolphinService;
import ru.yandex.travel.orders.workflows.orderitem.dolphin.IncorrectOrderStateException;
import ru.yandex.travel.orders.workflows.orderitem.dolphin.ManualTicketGenerator;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;

@Slf4j
@IgnoreEvents(types = {TMoneyAcquired.class})
public class ConfirmedStateHandler extends BaseDolphinHandler {
    public ConfirmedStateHandler(DolphinService dolphinService, Meters meters, ManualTicketGenerator ticketGenerator) {
        super(dolphinService, meters, ticketGenerator);
    }

    @HandleEvent
    public void scheduleRefund(TRefundStart message, StateContext<EDolphinItemState, DolphinOrderItem> context) {
        context.setState(EDolphinItemState.IS_REFUNDING);
        context.scheduleEvent(TRefundCommit.newBuilder()
                .setToken(message.getToken())
                .setForce(message.getForce())
                .setReason(message.getReason())
                .setSkipFinEvents(message.getSkipFinEvents())
                .setMoneyRefundMode(message.getMoneyRefundMode())
                .setRefundDescription(message.getRefundDescription())
                .build());
        log.info("scheduled refund");
    }

    @HandleEvent
    public void handleRefresh(TRefresh message, StateContext<EDolphinItemState, DolphinOrderItem> context) {
        String orderCode = context.getWorkflowEntity().getDolphinOrderCode();
        Preconditions.checkNotNull(orderCode, "Partner order code is not set");
        var order = dolphinService.get(orderCode, getCallContext(context.getWorkflowEntity(),
                CallContext.CallPhase.ORDER_REFUND));
        var itinerary = context.getWorkflowEntity().getItinerary();
        if (order.getState() != OrderState.OK) {
            throw new IncorrectOrderStateException(orderCode, order.getState());
        }
        checkAndUpdateGuests(order.getRequest().getGuests(), itinerary.getGuests());
    }
}
