package ru.yandex.travel.orders.workflows.orderitem.dolphin.handlers;

import java.util.UUID;
import java.util.stream.Collectors;

import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.common.partners.base.CallContext;
import ru.yandex.travel.hotels.common.partners.dolphin.model.CreateOrderRequest;
import ru.yandex.travel.hotels.common.partners.dolphin.model.Guest;
import ru.yandex.travel.hotels.common.partners.dolphin.model.Order;
import ru.yandex.travel.hotels.common.partners.dolphin.model.PriceKey;
import ru.yandex.travel.orders.entities.DolphinOrderItem;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.workflow.hotels.dolphin.proto.EDolphinItemState;
import ru.yandex.travel.orders.workflow.hotels.dolphin.proto.TConfirmationCommit;
import ru.yandex.travel.orders.workflow.hotels.dolphin.proto.TRefresh;
import ru.yandex.travel.orders.workflow.hotels.proto.TServiceInManualConfirmation;
import ru.yandex.travel.orders.workflows.orderitem.dolphin.DolphinService;
import ru.yandex.travel.orders.workflows.orderitem.dolphin.DuplicateDolphinOrderException;
import ru.yandex.travel.orders.workflows.orderitem.dolphin.ManualTicketGenerator;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
public class ConfirmingStateHandler extends BaseDolphinHandler {
    public ConfirmingStateHandler(DolphinService dolphinService, Meters meters, ManualTicketGenerator ticketGenerator) {
        super(dolphinService, meters, ticketGenerator);
    }

    @HandleEvent
    public void handleConfirmationCommit(TConfirmationCommit message,
                                         StateContext<EDolphinItemState, DolphinOrderItem> context) {
        var itinerary = context.getWorkflowEntity().getItinerary();
        var request = CreateOrderRequest.builder()
                .priceKey(PriceKey.builder()
                        .hotelId(itinerary.getHotelId())
                        .tourId(itinerary.getTourId())
                        .pansionId(itinerary.getPansionId())
                        .roomId(itinerary.getRoomId())
                        .roomCategoryId(itinerary.getRoomCatId())
                        .date(itinerary.getCheckinDate())
                        .nights(itinerary.getNights())
                        .beds(itinerary.getBeds())
                        .build())
                .adults(itinerary.getOccupancy().getAdults())
                .children(itinerary.getOccupancy().getChildren().stream().sorted((a, b) -> Integer.compare(b, a)).collect(Collectors.toList()))
                .email(itinerary.getCustomerEmail())
                .phone(itinerary.getCustomerPhone())
                .guests(
                        itinerary.getGuests()
                                .stream()
                                .filter(ru.yandex.travel.hotels.common.orders.Guest::hasFilledName)
                                .map(guest -> Guest.createCyrillic(guest.getFirstName(),
                                        guest.getLastName()))
                                .collect(Collectors.toList())
                ).build();

        Order order;
        try {
            order = dolphinService.findExistingOrder(request, getCallContext(context.getWorkflowEntity(),
                    CallContext.CallPhase.ORDER_CONFIRMATION));
        } catch (DuplicateDolphinOrderException exception) {
            UUID ticketId = ticketGenerator.generateDuplicateOrderTicket(context.getWorkflowEntity().getOrder(),
                    exception.getOtherOrderPrettyId(), exception.getOtherOrderErrorCode(), context);
            context.setState(EDolphinItemState.IS_MANUAL_CONFIRMATION);
            context.scheduleExternalEvent(context.getWorkflowEntity().getOrderWorkflowId(),
                    TServiceInManualConfirmation.newBuilder().build());
            context.getWorkflowEntity().getHotelItinerary().setManualTicketId(ticketId);
            return;
        }
        if (order == null) {
            order = dolphinService.createAndGet(request, getCallContext(context.getWorkflowEntity(),
                    CallContext.CallPhase.ORDER_CONFIRMATION));
        }
        context.getWorkflowEntity().setDolphinOrderCode(order.getCode());
        switch (order.getState()) {
            case OK:
                handleSuccessfulOrderCreation(order, itinerary, context);
                break;
            case WAIT_LIST:
            case IN_WORK:
                log.warn("Partner reports that the order is in {}", order.getState());
                context.setState(EDolphinItemState.IS_POLLING_FOR_STATUS);
                context.scheduleEvent(TRefresh.newBuilder().build());
                break;
            case REBOOKING:
                log.warn("Partner reports that the order is in {}", order.getState());
                UUID ticketId = ticketGenerator.generateWaitListTicket(context.getWorkflowEntity().getOrder(),
                        order.getCode(), itinerary.getOrderDetails(), context);
                context.setState(EDolphinItemState.IS_MANUAL_CONFIRMATION);
                context.getWorkflowEntity().getHotelItinerary().setManualTicketId(ticketId);
                context.scheduleExternalEvent(context.getWorkflowEntity().getOrderWorkflowId(),
                        TServiceInManualConfirmation.newBuilder().build());
                break;
            default:
                var msg = String.format("Itinerary '%s': Unexpected partner state '%s'",
                        itinerary.getServiceId(), order.getState());
                log.error(msg);
                throw new RuntimeException(msg);
        }
    }
}
