package ru.yandex.travel.orders.workflows.orderitem.dolphin.handlers;

import java.time.Instant;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.common.orders.CancellationDetails;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.orders.entities.DolphinOrderItem;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.services.partners.BillingPartnerService;
import ru.yandex.travel.orders.workflow.hotels.dolphin.proto.EDolphinItemState;
import ru.yandex.travel.orders.workflow.hotels.proto.TReservationStart;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.orders.workflow.order.proto.TServiceReserved;
import ru.yandex.travel.orders.workflows.orderitem.dolphin.DolphinFiscalItemCreator;
import ru.yandex.travel.orders.workflows.orderitem.dolphin.DolphinProperties;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@RequiredArgsConstructor
@Slf4j
public class NewStateHandler extends AnnotatedStatefulWorkflowEventHandler<EDolphinItemState, DolphinOrderItem> {
    private final BillingPartnerService billingPartnerService;
    private final DolphinProperties dolphinProperties;

    private final Meters meters;

    private final DolphinFiscalItemCreator fiscalItemCreator;

    @HandleEvent
    public void handleReservationStart(TReservationStart message,
                                       StateContext<EDolphinItemState, DolphinOrderItem> context) {
        handleGenericRegister(null, context);
    }

    @HandleEvent
    public void handleGenericRegister(ru.yandex.travel.orders.workflow.orderitem.generic.proto.TReservationStart message,
                                      StateContext<EDolphinItemState, DolphinOrderItem> context) {
        DolphinOrderItem orderItem = context.getWorkflowEntity();
        billingPartnerService.addAgreementOrThrow(orderItem);
        var itinerary = orderItem.getItinerary();
        if (dolphinProperties.getTestHotelsIds() != null && !dolphinProperties.getTestHotelsIds().contains(itinerary.getHotelId()) && orderItem.getTestContext() == null) {
            log.warn("Test hotel ids enabled and the hotel is not for testing!");
            context.setState(EDolphinItemState.IS_CANCELLED);
            meters.incrementCancellationCounter(EPartnerId.PI_DOLPHIN, CancellationDetails.Reason.BLACKLISTED);
            context.scheduleExternalEvent(orderItem.getOrderWorkflowId(),
                    TServiceCancelled.newBuilder().setServiceId(orderItem.getId().toString()).build());
        } else {
            // TODO(tivelkov): may be we need to call CalculateOrder to re-ensure that there are some places available
            log.info("Immediately reserving dolphin itinerary");
            fiscalItemCreator.addFiscalItems(orderItem);
            orderItem.setExpiresAt(Instant.now().plus(dolphinProperties.getReservationDuration()));
            context.setState(EDolphinItemState.IS_RESERVED);

            context.scheduleExternalEvent(orderItem.getOrderWorkflowId(),
                    TServiceReserved.newBuilder().setServiceId(orderItem.getId().toString()).build());

        }
    }
}
