package ru.yandex.travel.orders.workflows.orderitem.dolphin.handlers;

import java.util.UUID;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.common.orders.CancellationDetails;
import ru.yandex.travel.hotels.common.partners.base.CallContext;
import ru.yandex.travel.hotels.common.partners.dolphin.model.ExtendState;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.orders.entities.DolphinOrderItem;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.workflow.hotels.dolphin.proto.EDolphinItemState;
import ru.yandex.travel.orders.workflow.hotels.dolphin.proto.TRefresh;
import ru.yandex.travel.orders.workflow.hotels.proto.TServiceInManualConfirmation;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.orders.workflows.orderitem.dolphin.DolphinService;
import ru.yandex.travel.orders.workflows.orderitem.dolphin.ManualTicketGenerator;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.exceptions.RetryableException;

@Slf4j
public class PollingStateHandler extends BaseDolphinHandler {

    public PollingStateHandler(DolphinService dolphinService, Meters meters, ManualTicketGenerator ticketGenerator) {
        super(dolphinService, meters, ticketGenerator);
    }

    @HandleEvent
    public void handlerRefresh(TRefresh message,
                               StateContext<EDolphinItemState, DolphinOrderItem> context) {
        String orderCode = context.getWorkflowEntity().getDolphinOrderCode();
        Preconditions.checkNotNull(orderCode, "Partner order code is not set");
        var order = dolphinService.get(orderCode, getCallContext(context.getWorkflowEntity(),
                CallContext.CallPhase.ORDER_CONFIRMATION));
        var itinerary = context.getWorkflowEntity().getItinerary();
        switch (order.getState()) {
            case ANNULATED:
                log.info("partner order is ANNULLED, itinerary is CANCELLED");
                if (itinerary.getManualTicketId() != null) {
                    ticketGenerator.updateWaitlistTicket(itinerary.getManualTicketId(),
                            ManualTicketGenerator.ManualOutcome.ANNULLED, context);
                }
                context.setState(EDolphinItemState.IS_CANCELLED);
                meters.incrementCancellationCounter(EPartnerId.PI_DOLPHIN, CancellationDetails.Reason.HOLD_EXPIRED);
                context.scheduleExternalEvent(context.getWorkflowEntity().getOrderWorkflowId(),
                        TServiceCancelled.newBuilder().setServiceId(context.getWorkflowEntity().getId().toString()).build());
                break;
            case OK:
                handleSuccessfulOrderCreation(order, itinerary, context);
                break;
            case WAIT_LIST:
            case IN_WORK:
                if (context.getAttempt() <= dolphinService.getProperties().getPollingRetryAttempts() &&
                        (order.getExtendState() == null || order.getExtendState() != ExtendState.DOUBLE_BOOKING)) {
                    var msg = String.format("Itinerary '%s': partner order is still in %s on attempt %d, " +
                            "will wait more", itinerary.getServiceId(), order.getState(), context.getAttempt());
                    log.warn(msg);
                    throw new RetryableException(msg, dolphinService.getProperties().getPollingRetryDelay());
                } else {
                    if (itinerary.getManualTicketId() != null) {
                        ticketGenerator.updateWaitlistTicket(itinerary.getManualTicketId(),
                                ManualTicketGenerator.ManualOutcome.STILL_BROKEN, context);
                    } else {
                        UUID ticketID = ticketGenerator.generateWaitListTicket(context.getWorkflowEntity().getOrder(),
                                orderCode, itinerary.getOrderDetails(), context);
                        context.getWorkflowEntity().getHotelItinerary().setManualTicketId(ticketID);
                    }
                    context.setState(EDolphinItemState.IS_MANUAL_CONFIRMATION);
                    context.scheduleExternalEvent(context.getWorkflowEntity().getOrderWorkflowId(),
                            TServiceInManualConfirmation.newBuilder().build());
                }
                break;
            case REBOOKING:
                if (itinerary.getManualTicketId() != null) {
                    if (itinerary.getManualTicketId() != null) {
                        ticketGenerator.updateWaitlistTicket(itinerary.getManualTicketId(),
                                ManualTicketGenerator.ManualOutcome.STILL_BROKEN, context);
                    }
                } else {
                    UUID ticketID = ticketGenerator.generateWaitListTicket(context.getWorkflowEntity().getOrder(),
                            orderCode, itinerary.getOrderDetails(), context);
                    context.getWorkflowEntity().getHotelItinerary().setManualTicketId(ticketID);
                    break;
                }
                context.setState(EDolphinItemState.IS_MANUAL_CONFIRMATION);
                context.scheduleExternalEvent(context.getWorkflowEntity().getOrderWorkflowId(),
                        TServiceInManualConfirmation.newBuilder().build());
                break;
        }

    }
}
