package ru.yandex.travel.orders.workflows.orderitem.dolphin.handlers;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.common.orders.DolphinHotelItinerary;
import ru.yandex.travel.hotels.common.partners.base.CallContext;
import ru.yandex.travel.hotels.common.partners.base.exceptions.PartnerException;
import ru.yandex.travel.hotels.common.partners.base.exceptions.RetryableHttpException;
import ru.yandex.travel.hotels.common.partners.base.exceptions.RetryableIOException;
import ru.yandex.travel.hotels.common.partners.dolphin.model.AnnulateResult;
import ru.yandex.travel.hotels.common.partners.dolphin.model.ExtendState;
import ru.yandex.travel.hotels.common.partners.dolphin.model.Order;
import ru.yandex.travel.hotels.common.partners.dolphin.model.OrderState;
import ru.yandex.travel.orders.entities.DolphinOrderItem;
import ru.yandex.travel.orders.proto.THotelRefundToken;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.workflow.hotels.dolphin.proto.EDolphinItemState;
import ru.yandex.travel.orders.workflow.hotels.dolphin.proto.TRefundCommit;
import ru.yandex.travel.orders.workflows.orderitem.RefundingUtils;
import ru.yandex.travel.orders.workflows.orderitem.dolphin.DolphinService;
import ru.yandex.travel.orders.workflows.orderitem.dolphin.ManualTicketGenerator;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.exceptions.RetryableException;

@Slf4j
public class RefundingStateHandler extends BaseDolphinHandler {

    public RefundingStateHandler(DolphinService dolphinService, Meters meters, ManualTicketGenerator ticketGenerator) {
        super(dolphinService, meters, ticketGenerator);
    }

    @HandleEvent
    public void handleRefundCommit(TRefundCommit message, StateContext<EDolphinItemState, DolphinOrderItem> context) {
        DolphinHotelItinerary itinerary = context.getWorkflowEntity().getItinerary();
        Preconditions.checkArgument(itinerary.getServiceId().equals(context.getWorkflowEntity().getId().toString()));
        Preconditions.checkNotNull(context.getWorkflowEntity().getDolphinOrderCode());
        Preconditions.checkArgument(context.getWorkflowEntity().getFiscalItems().size() == 1,
                "Unexpected number of fiscal items");
        log.info("calling Dolphin refund API");
        try {
            AnnulateResult res = dolphinService.annulateOrder(context.getWorkflowEntity().getDolphinOrderCode(),
                    getCallContext(context.getWorkflowEntity(), CallContext.CallPhase.ORDER_REFUND));
            if (!res.isSuccess()) {
                Order order = dolphinService.get(context.getWorkflowEntity().getDolphinOrderCode(),
                        getCallContext(context.getWorkflowEntity(), CallContext.CallPhase.ORDER_REFUND));
                if (order.getState() != OrderState.ANNULATED) {
                    if (order.getState() == OrderState.OK && order.getExtendState() == ExtendState.PENALTY) {
                        if (message.getForce()) {
                            log.warn("Dolphin returns Penalty, force is set, so committing refund");
                        } else {
                            throw new RuntimeException("Failed to annulate order " +
                                    context.getWorkflowEntity().getDolphinOrderCode() +
                                    ", Dolphin reports penalty, force is not set");
                        }
                    } else {
                        throw new RuntimeException("Failed to annulate order " +
                                context.getWorkflowEntity().getDolphinOrderCode() +
                                ", current order state is " + order.getState());
                    }
                }
            }
        } catch (PartnerException exception) {
            Throwable cause = exception.getCause();
            if (cause instanceof RetryableHttpException) {
                throw new RetryableException(cause);
            } else if (cause instanceof RetryableIOException) {
                throw new RetryableException(cause);
            } else {
                throw new RuntimeException(cause);
            }
        }
        THotelRefundToken refundToken = RefundingUtils.getRefundToken(message.getToken());
        itinerary.setRefundInfo(RefundingUtils.createRefundInfo(refundToken, message.getReason()));
        context.setState(EDolphinItemState.IS_REFUNDED);
        RefundingUtils.scheduleServiceRefundedEvent(context, refundToken,
                message.getSkipFinEvents(), message.getMoneyRefundMode(), message.getRefundDescription());
        log.info("REFUNDED");
    }
}
