package ru.yandex.travel.orders.workflows.orderitem.dolphin.handlers;

import lombok.extern.slf4j.Slf4j;
import org.apache.logging.log4j.util.Strings;

import ru.yandex.travel.hotels.common.orders.CancellationDetails;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.orders.entities.DolphinOrderItem;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.workflow.hotels.dolphin.proto.EDolphinItemState;
import ru.yandex.travel.orders.workflow.hotels.dolphin.proto.TConfirmationCommit;
import ru.yandex.travel.orders.workflow.hotels.dolphin.proto.TReservationExpired;
import ru.yandex.travel.orders.workflow.hotels.proto.TCancellationStart;
import ru.yandex.travel.orders.workflow.hotels.proto.TConfirmationStart;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.orders.workflows.orderitem.dolphin.DolphinService;
import ru.yandex.travel.orders.workflows.orderitem.dolphin.ManualTicketGenerator;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
public class ReservedStateHandler extends BaseDolphinHandler {
    public ReservedStateHandler(DolphinService dolphinService, Meters meters, ManualTicketGenerator ticketGenerator) {
        super(dolphinService, meters, ticketGenerator);
    }

    @HandleEvent
    public void startCancellation(TCancellationStart message,
                                  StateContext<EDolphinItemState, DolphinOrderItem> context) {
        context.setState(EDolphinItemState.IS_CANCELLED);
        context.scheduleExternalEvent(context.getWorkflowEntity().getOrderWorkflowId(),
                TServiceCancelled.newBuilder().setServiceId(context.getWorkflowEntity().getId().toString()).build());
        if (Strings.isNotEmpty(message.getReason())) {
            var itinerary = context.getWorkflowEntity().getItinerary();
            itinerary.setOrderCancellationDetails(CancellationDetails.create(message.getReason()));
            meters.incrementCancellationCounter(EPartnerId.PI_DOLPHIN, message.getReason());
        }
        if (context.getWorkflowEntity().getHotelItinerary().getManualTicketId() != null) {
            ticketGenerator.updateWaitlistTicket(context.getWorkflowEntity().getHotelItinerary().getManualTicketId(),
                    ManualTicketGenerator.ManualOutcome.CANCELLED_WITH_NO_REMOTE, context);
        }
        log.info("Immediately cancelled");
    }

    @HandleEvent
    public void startGenericCancellation(ru.yandex.travel.orders.workflow.orderitem.generic.proto.TCancellationStart message,
                                         StateContext<EDolphinItemState, DolphinOrderItem> context) {
        startCancellation(TCancellationStart.newBuilder().build(), context);
    }

    @HandleEvent
    public void handleReservationExpired(TReservationExpired message,
                                         StateContext<EDolphinItemState, DolphinOrderItem> context) {
        context.setState(EDolphinItemState.IS_CANCELLED);
        context.scheduleExternalEvent(context.getWorkflowEntity().getOrderWorkflowId(),
                TServiceCancelled.newBuilder().setServiceId(context.getWorkflowEntity().getId().toString()).build());
        log.info("Reservation expired. Scheduled cancellation");
        var itinerary = context.getWorkflowEntity().getItinerary();
        var reason = CancellationDetails.Reason.MONEY_ACQUIRE_ERROR;
        itinerary.setOrderCancellationDetails(CancellationDetails.create(reason));
        meters.incrementCancellationCounter(EPartnerId.PI_DOLPHIN, reason);
    }

    @HandleEvent
    public void startConfirmation(TConfirmationStart message,
                                  StateContext<EDolphinItemState, DolphinOrderItem> context) {
        startGenericConfirmation(null, context);
    }

    @HandleEvent
    public void startGenericConfirmation(ru.yandex.travel.orders.workflow.orderitem.generic.proto.TConfirmationStart message,
                                         StateContext<EDolphinItemState, DolphinOrderItem> context) {
        context.setState(EDolphinItemState.IS_CONFIRMING);
        context.getWorkflowEntity().getItinerary().setServiceId(context.getWorkflowEntity().getId().toString());
        context.scheduleEvent(TConfirmationCommit.newBuilder().build());
        log.info("Scheduled confirmation");
    }
}
