package ru.yandex.travel.orders.workflows.orderitem.expedia.handlers;

import java.util.Map;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.common.orders.CancellationDetails;
import ru.yandex.travel.hotels.common.orders.ExpediaHotelItinerary;
import ru.yandex.travel.hotels.common.partners.base.exceptions.PartnerException;
import ru.yandex.travel.hotels.common.partners.base.exceptions.RetryableHttpException;
import ru.yandex.travel.hotels.common.partners.base.exceptions.RetryableIOException;
import ru.yandex.travel.hotels.common.partners.expedia.ExpediaClient;
import ru.yandex.travel.hotels.common.partners.expedia.Helpers;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.Itinerary;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.ItineraryRoom;
import ru.yandex.travel.hotels.common.partners.expedia.model.common.Error;
import ru.yandex.travel.hotels.common.partners.expedia.model.common.ErrorField;
import ru.yandex.travel.orders.entities.ExpediaOrderItem;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.workflow.hotels.expedia.proto.EExpediaItemState;
import ru.yandex.travel.orders.workflows.orderitem.expedia.ExpediaProperties;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.exceptions.RetryableException;

@Slf4j
public class BaseExpediaHandler extends AnnotatedStatefulWorkflowEventHandler<EExpediaItemState, ExpediaOrderItem> {
    private final static Map<String, CancellationDetails.ValidatedFieldType> VALIDATED_FIELD_MAP = Map.of(
            "family_name", CancellationDetails.ValidatedFieldType.LAST_NAME,
            "given_name", CancellationDetails.ValidatedFieldType.FIRST_NAME,
            "phone", CancellationDetails.ValidatedFieldType.PHONE,
            "number", CancellationDetails.ValidatedFieldType.PHONE,
            "country_code", CancellationDetails.ValidatedFieldType.COUNTRY_CODE,
            "email", CancellationDetails.ValidatedFieldType.EMAIL
    );
    protected final Meters meters;
    protected ExpediaClient expediaClient;
    protected ExpediaProperties expediaProperties;

    protected BaseExpediaHandler(ExpediaClient expediaClient, ExpediaProperties expediaProperties, Meters meters) {
        super();
        this.expediaClient = expediaClient;
        this.expediaProperties = expediaProperties;
        this.meters = meters;
    }

    protected <T> T wrap(Supplier<T> supplier) {
        try {
            return supplier.get();
        } catch (PartnerException exception) {
            Throwable cause = exception.getCause();
            if (cause instanceof RetryableHttpException) {
                throw new RetryableException(cause);
            } else if (cause instanceof RetryableIOException) {
                throw new RetryableException(cause);
            } else {
                throw exception;
            }
        }
    }

    public void checkApiVersionAndUpdateIfNeeded(ExpediaHotelItinerary itinerary) {
        if (itinerary.getApiVersion() != expediaClient.getApiVersion()) {
            log.warn("Itinerary has an API version of {}, current default is {}, will update",
                    itinerary.getApiVersion().getValue(), expediaClient.getApiVersion().getValue());
            Itinerary existing = wrap(() -> expediaClient.getItineraryByAffiliateIdSync(itinerary.getAffiliateId(),
                    itinerary.getCustomerEmail(), itinerary.getCustomerIp(), itinerary.getCustomerUserAgent(),
                    itinerary.getCustomerSessionId()));
            if (existing == null) {
                log.error("Unable to update the API version as the itinerary could not be found");
                return;
            }
            if (existing.getLinks() != null) {
                String confirmationToken = Helpers.retrieveConfirmationToken(existing.getLinks().getResume().getHref(),
                        itinerary.getExpediaItineraryId());
                itinerary.setExpediaConfirmationToken(confirmationToken);
                itinerary.setApiVersion(expediaClient.getApiVersion());
                log.info("Update completed: API version set to {}, confirmation token updated",
                        expediaClient.getApiVersion());
            }
            if (existing.getRooms() != null && existing.getRooms().size() > 0) {
                extractRefundInfo(itinerary, existing.getRooms().get(0));
                itinerary.setApiVersion(expediaClient.getApiVersion());
                log.info("Update completed: API version set to {}, refund token and info updated",
                        expediaClient.getApiVersion());
            }
        }
    }

    protected CancellationDetails errorToCancellationDetails(Error error) {
        switch (error.getType()) {
            case Error.PRICE_MISMATCH:
                return CancellationDetails.create(CancellationDetails.Reason.PRICE_CHANGED);
            case Error.ROOMS_UNAVAILABLE:
                return CancellationDetails.create(CancellationDetails.Reason.SOLD_OUT);
            case Error.INVALID_INPUT:
                if (error.getErrors().get(0).getType().equals(Error.DUPLICATE_ITINERARY)) {
                    return CancellationDetails.create(CancellationDetails.Reason.DUPLICATE);
                } else {
                    CancellationDetails details =
                            CancellationDetails.create(CancellationDetails.Reason.INVALID_INPUT);
                    var fieldNames = error.getErrors().stream()
                            .map(Error::getFields)
                            .flatMap(ef -> ef.stream()
                                    .map(ErrorField::getName))
                            .collect(Collectors.toList());
                    fieldNames.forEach(fn -> {
                        if (!VALIDATED_FIELD_MAP.containsKey(fn)) {
                            throw new RuntimeException(String.format("Unexpected invalid field '%s' for " +
                                    "'invalid_field' " +
                                    "error in '%s' call", fn, this.getClass().getSimpleName()));
                        }
                        var field = new CancellationDetails.InvalidField();
                        field.setFieldType(VALIDATED_FIELD_MAP.get(fn));
                        if (field.getFieldType() == CancellationDetails.ValidatedFieldType.FIRST_NAME || field.getFieldType() == CancellationDetails.ValidatedFieldType.LAST_NAME) {
                            field.setGuestIndex(0);
                        }
                        details.getInvalidInputDetails().add(field);
                    });
                    return details;
                }
            default:
                return null;
        }
    }

    protected void extractRefundInfo(ExpediaHotelItinerary itinerary, ItineraryRoom itineraryRoom) {
        if (itineraryRoom.getLinks() != null) {
            String[] refundInfo = Helpers.retriveRoomAndRefundToken(itineraryRoom.getLinks().getCancel().getHref(),
                    itinerary.getExpediaItineraryId());
            Preconditions.checkNotNull(refundInfo);
            Preconditions.checkArgument(refundInfo.length == 2);
            itinerary.setRefundId(refundInfo[0]);
            itinerary.setExpediaRefundToken(refundInfo[1]);
        } else {
            itinerary.setExpediaRefundToken(null);
        }
    }
}
