package ru.yandex.travel.orders.workflows.orderitem.expedia.handlers;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.common.orders.CancellationDetails;
import ru.yandex.travel.hotels.common.orders.ExpediaHotelItinerary;
import ru.yandex.travel.hotels.common.partners.expedia.ExpediaClient;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.CancellationStatus;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.orders.entities.ExpediaOrderItem;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.workflow.hotels.expedia.proto.EExpediaItemState;
import ru.yandex.travel.orders.workflow.hotels.expedia.proto.TCancellationCommit;
import ru.yandex.travel.orders.workflow.hotels.expedia.proto.TReservationExpired;
import ru.yandex.travel.orders.workflow.hotels.proto.TCancellationStart;
import ru.yandex.travel.orders.workflow.hotels.proto.TConfirmationStart;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.orders.workflows.orderitem.expedia.ExpediaProperties;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;
import ru.yandex.travel.workflow.exceptions.RetryableException;

@Slf4j
@IgnoreEvents(types = {TCancellationStart.class, TConfirmationStart.class, TReservationExpired.class})
public class CancellingStateHandler extends BaseExpediaHandler {

    public CancellingStateHandler(ExpediaClient expediaClient, ExpediaProperties expediaProperties, Meters meters) {
        super(expediaClient, expediaProperties, meters);
    }

    @HandleEvent(TCancellationCommit.class)
    public void doCancellation(TCancellationCommit message, StateContext<EExpediaItemState, ExpediaOrderItem> context) {
        ExpediaHotelItinerary itinerary = context.getWorkflowEntity().getItinerary();
        Preconditions.checkArgument(itinerary.getServiceId().equals(context.getWorkflowEntity().getId().toString()));
        Preconditions.checkNotNull(itinerary.getExpediaItineraryId());
        Preconditions.checkNotNull(itinerary.getExpediaConfirmationToken());
        checkApiVersionAndUpdateIfNeeded(itinerary);
        log.info("Calling Expedia cancellation API");

        CancellationStatus cancellationStatus =
                wrap(() -> expediaClient.cancelItinerarySync(itinerary.getExpediaItineraryId(),
                itinerary.getExpediaConfirmationToken(),
                itinerary.getCustomerIp(), itinerary.getCustomerUserAgent(), itinerary.getCustomerSessionId()));
        switch (cancellationStatus) {
            case UNKNOWN:
                throw new RetryableException("Expedia could not confirm itinerary cancellation");
            case NOT_FOUND:
                log.warn("404 on cancellation request, probably already cancelled or expired");
                context.setState(EExpediaItemState.IS_CANCELLED);
                meters.incrementCancellationCounter(EPartnerId.PI_EXPEDIA, CancellationDetails.Reason.HOLD_EXPIRED);
                context.scheduleExternalEvent(context.getWorkflowEntity().getOrderWorkflowId(),
                        TServiceCancelled.newBuilder().setServiceId(context.getWorkflowEntity().getId().toString()).build());
                log.info("CANCELLED");
                return;
            case SUCCESS:
                log.info("Cancellation call successful, verifying");
                var cancelled = wrap(() -> expediaClient.getItinerarySync(itinerary.getExpediaItineraryId(),
                        itinerary.getExpediaConfirmationToken(), itinerary.getCustomerIp(),
                        itinerary.getCustomerUserAgent(), itinerary.getCustomerSessionId()));
                Preconditions.checkArgument(cancelled == null,
                        "Itinerary it supposed to be cancelled, but seems like it was not");
                context.setState(EExpediaItemState.IS_CANCELLED);
                meters.incrementCancellationCounter(EPartnerId.PI_EXPEDIA, CancellationDetails.Reason.FX_RATE_EXPIRED);
                context.scheduleExternalEvent(context.getWorkflowEntity().getOrderWorkflowId(),
                        TServiceCancelled.newBuilder().setServiceId(context.getWorkflowEntity().getId().toString()).build());
                log.info("CANCELLED");
        }
    }
}
