package ru.yandex.travel.orders.workflows.orderitem.expedia.handlers;

import java.util.Optional;
import java.util.Set;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.common.orders.CancellationDetails;
import ru.yandex.travel.hotels.common.orders.ExpediaHotelItinerary;
import ru.yandex.travel.hotels.common.partners.expedia.ExpediaClient;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.Itinerary;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.ItineraryRoom;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.ResumeReservationStatus;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.RoomStatus;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.orders.entities.ExpediaOrderItem;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.workflow.hotels.expedia.proto.EExpediaItemState;
import ru.yandex.travel.orders.workflow.hotels.expedia.proto.TConfirmationCommit;
import ru.yandex.travel.orders.workflow.hotels.expedia.proto.TRefresh;
import ru.yandex.travel.orders.workflow.hotels.expedia.proto.TStartPolling;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.orders.workflows.orderitem.expedia.ExpediaProperties;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
public class ConfirmingStateHandler extends BaseExpediaHandler {
    public ConfirmingStateHandler(ExpediaClient expediaClient, ExpediaProperties expediaProperties, Meters meters) {
        super(expediaClient, expediaProperties, meters);
    }

    @HandleEvent
    public void doConfirmation(TConfirmationCommit message, StateContext<EExpediaItemState, ExpediaOrderItem> context) {
        ExpediaHotelItinerary itinerary = context.getWorkflowEntity().getItinerary();
        Preconditions.checkArgument(itinerary.getServiceId().equals(context.getWorkflowEntity().getId().toString()));
        Preconditions.checkNotNull(itinerary.getExpediaItineraryId());
        Preconditions.checkNotNull(itinerary.getExpediaConfirmationToken());
        checkApiVersionAndUpdateIfNeeded(itinerary);
        log.info("fetching data from Expedia API");
        Itinerary result = wrap(() -> expediaClient.getItinerarySync(itinerary.getExpediaItineraryId(),
                itinerary.getExpediaConfirmationToken(),
                itinerary.getCustomerIp(), itinerary.getCustomerUserAgent(), itinerary.getCustomerSessionId()));
        if (result == null) {
            log.warn("no hold to resume found, probably expired");
            itinerary.setOrderCancellationDetails(CancellationDetails.create(CancellationDetails.Reason.HOLD_EXPIRED));
            context.setState(EExpediaItemState.IS_CANCELLED);
            meters.incrementCancellationCounter(EPartnerId.PI_EXPEDIA, CancellationDetails.Reason.HOLD_EXPIRED);
            context.scheduleExternalEvent(context.getWorkflowEntity().getOrderWorkflowId(),
                    TServiceCancelled.newBuilder().setServiceId(context.getWorkflowEntity().getId().toString()).build());
            return;
        }
        Optional<ItineraryRoom> itineraryRoom = result.getRooms() != null ?
                result.getRooms().stream().filter(room -> room.getId().equals(itinerary.getRoomId())).findFirst() :
                Optional.empty();
        if (itineraryRoom.isPresent()) {
            log.info("Itinerary room already exists, this may happen only on retry. Will do polling without " +
                    "attempt to resume");
            Preconditions.checkArgument(
                    Set.of(RoomStatus.PENDING, RoomStatus.BOOKED).contains(itineraryRoom.get().getStatus()),
                    "Unexpected room status " + itineraryRoom.get().getStatus().getValue());
            context.setState(EExpediaItemState.IS_POLLING_FOR_ITINERARY);
            // this may happen only while retrying after failed resume call. As we do not know if the resume call
            // has been confirmed (204) or delayed (202), let's assume the strict - and do not allow missing
            // itineraries while polling, so we pass allowMissingItinerary=false to the message
            context.scheduleEvent(TStartPolling.newBuilder().setAllowMissingItinerary(false).build());
            return;
        } else {
            // no room data, but itinerary is present. Need to call resume to confirm it
            log.info("calling Expedia confirmation API");
            ResumeReservationStatus resumeStatus =
                    wrap(() -> expediaClient.resumeItinerarySync(itinerary.getExpediaItineraryId(),
                            itinerary.getExpediaConfirmationToken(), itinerary.getCustomerIp(),
                            itinerary.getCustomerUserAgent(), itinerary.getCustomerSessionId()));
            switch (resumeStatus) {
                case SUCCESS:
                    log.info("confirmation confirmed inplace, switching to polling to get confirmation info");
                    context.setState(EExpediaItemState.IS_POLLING_FOR_ITINERARY);
                    context.scheduleEvent(TStartPolling.newBuilder().setAllowMissingItinerary(false).build());
                    return;
                case UNKNOWN:
                    log.info("confirmation state is unknown, switching to polling to get to get the state and " +
                            "confirmation info");
                    context.setState(EExpediaItemState.IS_POLLING_FOR_ITINERARY);
                    context.scheduleEvent(TStartPolling.newBuilder().setAllowMissingItinerary(true).build());
                    return;
                case NOT_FOUND:
                case ROLLED_BACK:
                    log.warn("no hold to resume found, probably expired");
                    itinerary.setOrderCancellationDetails(CancellationDetails.create(CancellationDetails.Reason.HOLD_EXPIRED));
                    context.setState(EExpediaItemState.IS_CANCELLED);
                    meters.incrementCancellationCounter(EPartnerId.PI_EXPEDIA,
                            CancellationDetails.Reason.HOLD_EXPIRED);
                    context.scheduleExternalEvent(context.getWorkflowEntity().getOrderWorkflowId(),
                            TServiceCancelled.newBuilder().setServiceId(context.getWorkflowEntity().getId().toString()).build());
                    return;
                case DUPLICATE:
                    log.warn("already resumed on resume call");
                    context.setState(EExpediaItemState.IS_POLLING_FOR_ITINERARY);
                    context.scheduleEvent(TStartPolling.newBuilder().setAllowMissingItinerary(false).build());
                    return;
            }
        }
    }

    @HandleEvent
    public void doRefresh(TRefresh message, StateContext<EExpediaItemState, ExpediaOrderItem> context) {
        log.warn("Received a deprecated `TRefresh` message. Falling back to an updated refresh-and-confirm routine");
        doConfirmation(TConfirmationCommit.newBuilder().build(), context);
    }
}
