package ru.yandex.travel.orders.workflows.orderitem.expedia.handlers;

import java.time.Duration;
import java.time.Instant;
import java.util.Optional;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.common.orders.CancellationDetails;
import ru.yandex.travel.hotels.common.orders.ConfirmationInfo;
import ru.yandex.travel.hotels.common.orders.ExpediaHotelItinerary;
import ru.yandex.travel.hotels.common.partners.expedia.ExpediaClient;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.Itinerary;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.ItineraryRoom;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.RoomStatus;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.orders.entities.ExpediaOrderItem;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.workflow.hotels.expedia.proto.EExpediaItemState;
import ru.yandex.travel.orders.workflow.hotels.expedia.proto.TStartPolling;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.orders.workflow.order.proto.TServiceConfirmed;
import ru.yandex.travel.orders.workflows.orderitem.expedia.ExpediaProperties;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.exceptions.RetryableException;

@Slf4j
public class PollingStateHandler extends BaseExpediaHandler {
    public PollingStateHandler(ExpediaClient expediaClient, ExpediaProperties expediaProperties, Meters meters) {
        super(expediaClient, expediaProperties, meters);
    }

    @HandleEvent
    public void doPoll(TStartPolling message, StateContext<EExpediaItemState, ExpediaOrderItem> context) throws InterruptedException {
        ExpediaHotelItinerary itinerary = context.getWorkflowEntity().getItinerary();
        Preconditions.checkArgument(itinerary.getServiceId().equals(context.getWorkflowEntity().getId().toString()));
        Preconditions.checkNotNull(itinerary.getExpediaItineraryId());
        Preconditions.checkNotNull(itinerary.getExpediaConfirmationToken());
        checkApiVersionAndUpdateIfNeeded(itinerary);
        log.info("polling data from Expedia API");
        Itinerary result = wrap(() -> expediaClient.getItinerarySync(itinerary.getExpediaItineraryId(),
                itinerary.getExpediaConfirmationToken(),
                itinerary.getCustomerIp(), itinerary.getCustomerUserAgent(), itinerary.getCustomerSessionId()));
        if (result == null) {
            if (message.getAllowMissingItinerary()) {
                log.warn("no hold to resume found, probably expired");
                itinerary.setOrderCancellationDetails(CancellationDetails.create(CancellationDetails.Reason.HOLD_EXPIRED));
                context.setState(EExpediaItemState.IS_CANCELLED);
                meters.incrementCancellationCounter(EPartnerId.PI_EXPEDIA, CancellationDetails.Reason.HOLD_EXPIRED);
                context.scheduleExternalEvent(context.getWorkflowEntity().getOrderWorkflowId(),
                        TServiceCancelled.newBuilder().setServiceId(context.getWorkflowEntity().getId().toString()).build());
            } else {
                throw new RetryableException("No itinerary on polling. Will try again");
            }
        } else {
            Optional<ItineraryRoom> itineraryRoom = result.getRooms() != null ?
                    result.getRooms().stream().filter(room -> room.getId().equals(itinerary.getRoomId())).findFirst() :
                    Optional.empty();
            if (itineraryRoom.isEmpty()) {
                throw new RetryableException("No itinerary room on polling", Duration.ofSeconds(40)); // Current 100 attempts * 40 sec is ~ 1 hour
            }
            if (itineraryRoom.get().getStatus() == RoomStatus.PENDING) {
                throw new RetryableException("Room is still pending, will retry");
            } else if (itineraryRoom.get().getStatus() != RoomStatus.BOOKED) {
                throw new RuntimeException("Unexpected room status in confirmed itinerary");
            }
            extractRefundInfo(itinerary, itineraryRoom.get());
            ConfirmationInfo confirmationInfo = new ConfirmationInfo();
            confirmationInfo.setPartnerConfirmationId(itineraryRoom.get().getConfirmationId().getExpedia());
            confirmationInfo.setHotelConfirmationId(itineraryRoom.get().getConfirmationId().getProperty());
            itinerary.setConfirmation(confirmationInfo);
            context.getWorkflowEntity().setConfirmedAt(Instant.now());
            context.setState(EExpediaItemState.IS_CONFIRMED);
            meters.incrementConfirmationCounter(EPartnerId.PI_EXPEDIA);
            context.scheduleExternalEvent(context.getWorkflowEntity().getOrderWorkflowId(),
                    TServiceConfirmed.newBuilder().setServiceId(context.getWorkflowEntity().getId().toString()).build());
            log.info("confirmation info received, itinerary is CONFIRMED");
        }
    }
}
