package ru.yandex.travel.orders.workflows.orderitem.expedia.handlers;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.common.orders.ExpediaHotelItinerary;
import ru.yandex.travel.hotels.common.partners.expedia.ExpediaClient;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.CancellationStatus;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.Itinerary;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.RoomStatus;
import ru.yandex.travel.orders.entities.ExpediaOrderItem;
import ru.yandex.travel.orders.proto.THotelRefundToken;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.workflow.hotels.expedia.proto.EExpediaItemState;
import ru.yandex.travel.orders.workflow.hotels.expedia.proto.TRefundCommit;
import ru.yandex.travel.orders.workflows.orderitem.RefundingUtils;
import ru.yandex.travel.orders.workflows.orderitem.expedia.ExpediaProperties;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.exceptions.RetryableException;

@Slf4j
public class RefundingStateHandler extends BaseExpediaHandler {
    public RefundingStateHandler(ExpediaClient expediaClient, ExpediaProperties expediaProperties, Meters meters) {
        super(expediaClient, expediaProperties, meters);
    }

    @HandleEvent
    public void doRefund(TRefundCommit message, StateContext<EExpediaItemState, ExpediaOrderItem> context) {
        ExpediaHotelItinerary itinerary = context.getWorkflowEntity().getItinerary();
        Preconditions.checkArgument(itinerary.getServiceId().equals(context.getWorkflowEntity().getId().toString()));
        Preconditions.checkNotNull(itinerary.getExpediaItineraryId());
        Preconditions.checkNotNull(itinerary.getExpediaRefundToken());
        Preconditions.checkNotNull(itinerary.getRefundId());
        checkApiVersionAndUpdateIfNeeded(itinerary);

        log.info("Getting itinerary to check room status");
        Itinerary result = wrap(() -> expediaClient.getItinerarySync(itinerary.getExpediaItineraryId(),
                itinerary.getExpediaConfirmationToken(),
                itinerary.getCustomerIp(), itinerary.getCustomerUserAgent(), itinerary.getCustomerSessionId()));

        boolean needToCancel = true;
        if (result.getRooms() != null) {
            var allCancelled = result.getRooms().stream()
                    .filter(room -> room.getId().equals(itinerary.getRoomId()))
                    .allMatch(x -> x.getStatus() == RoomStatus.CANCELED);
            if (allCancelled) {
                log.info("Rooms are already cancelled, nothing to do");
                needToCancel = false;
            }
        }

        if (needToCancel) {
            log.info("calling Expedia refund API");
            CancellationStatus cancellationStatus =
                    wrap(() -> expediaClient.cancelConfirmedItinerarySync(itinerary.getExpediaItineraryId(),
                            itinerary.getExpediaRefundToken(), itinerary.getRefundId(),
                            itinerary.getCustomerIp(), itinerary.getCustomerUserAgent(),
                            itinerary.getCustomerSessionId()));
            switch (cancellationStatus) {
                case UNKNOWN:
                    throw new RetryableException("Expedia could not confirm itinerary cancellation");
                case ALREADY_CANCELLED:
                    log.warn("400 (room_already_cancelled) on refund request, seems like already refunded");
                    break;
                case POST_CHECKIN_CANCEL:
                case POST_CHECKOUT_CANCEL:
                    String msg = "Post-checkin or post-checkout cancellation reported by expedia";
                    if (message.getForce()) {
                        log.warn(msg);
                        break;
                    } else {
                        throw new RuntimeException(msg);
                    }
            }
        }

        THotelRefundToken refundToken = RefundingUtils.getRefundToken(message.getToken());
        itinerary.setRefundInfo(RefundingUtils.createRefundInfo(refundToken, message.getReason()));
        context.setState(EExpediaItemState.IS_REFUNDED);
        RefundingUtils.scheduleServiceRefundedEvent(context, refundToken,
                message.getSkipFinEvents(), message.getMoneyRefundMode(), message.getRefundDescription());
        log.info("REFUNDED");
    }
}
