package ru.yandex.travel.orders.workflows.orderitem.expedia.handlers;

import java.time.Instant;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.common.orders.CancellationDetails;
import ru.yandex.travel.hotels.common.orders.ExpediaHotelItinerary;
import ru.yandex.travel.hotels.common.partners.expedia.ExpediaClient;
import ru.yandex.travel.hotels.common.partners.expedia.Helpers;
import ru.yandex.travel.hotels.common.partners.expedia.exceptions.ErrorException;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.Itinerary;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.ItineraryReservationRequest;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.ReservationResult;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.orders.entities.ExpediaOrderItem;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.workflow.hotels.expedia.proto.EExpediaItemState;
import ru.yandex.travel.orders.workflow.hotels.expedia.proto.TReservationCommit;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.orders.workflow.order.proto.TServiceReserved;
import ru.yandex.travel.orders.workflows.orderitem.expedia.ExpediaFiscalItemCreator;
import ru.yandex.travel.orders.workflows.orderitem.expedia.ExpediaProperties;
import ru.yandex.travel.orders.workflows.orderitem.expedia.UnexpectedItineraryDuplicateException;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
public class ReservingStateHandler extends BaseExpediaHandler {
    private final ExpediaFiscalItemCreator fiscalItemCreator;

    public ReservingStateHandler(ExpediaClient expediaClient, ExpediaProperties expediaProperties, Meters meters,
                                 ExpediaFiscalItemCreator expediaFiscalItemCreator) {
        super(expediaClient, expediaProperties, meters);
        this.fiscalItemCreator = expediaFiscalItemCreator;
    }

    @HandleEvent(TReservationCommit.class)
    public void doExpediaReservation(TReservationCommit message,
                                     StateContext<EExpediaItemState, ExpediaOrderItem> context) {
        ExpediaHotelItinerary itinerary = context.getWorkflowEntity().getItinerary();
        ExpediaOrderItem item = context.getWorkflowEntity();
        Preconditions.checkNotNull(itinerary.getExpediaReservationToken());
        itinerary.setServiceId(item.getId().toString());
        if (itinerary.getExpiresAtInstant().isBefore(Instant.now())) {
            log.info("Exchange rate has expired");
            log.info("CANCELLED");
            itinerary.setOrderCancellationDetails(CancellationDetails.create(CancellationDetails.Reason.FX_RATE_EXPIRED));
            context.setState(EExpediaItemState.IS_CANCELLED);
            meters.incrementCancellationCounter(EPartnerId.PI_EXPEDIA, CancellationDetails.Reason.FX_RATE_EXPIRED);
            context.scheduleExternalEvent(context.getWorkflowEntity().getOrderWorkflowId(),
                    TServiceCancelled.newBuilder().setServiceId(context.getWorkflowEntity().getId().toString()).build());
            return;
        }
        Itinerary existing =
                wrap(() -> expediaClient.getItineraryByAffiliateIdSync(itinerary.getAffiliateId(),
                        itinerary.getCustomerEmail(),
                        itinerary.getCustomerIp(),
                        itinerary.getCustomerUserAgent(),
                        itinerary.getCustomerSessionId()));
        String expediaItineraryId;
        String confirmationToken;
        if (existing != null) {
            log.warn("Found an already existing itinerary for affiliate id '{}', probably duplicate",
                    itinerary.getAffiliateId());
            expediaItineraryId = existing.getItineraryId();
            if (expediaClient.getApiVersion() != itinerary.getApiVersion()) {
                log.warn("Itinerary had an API version of {}, but was re-fetched using {} during deduplication, " +
                                "updating stored version",
                        itinerary.getApiVersion().getValue(), expediaClient.getApiVersion().getValue());
                itinerary.setApiVersion(expediaClient.getApiVersion());
            }
            confirmationToken = Helpers.retrieveConfirmationToken(existing.getLinks().getResume().getHref(),
                    existing.getItineraryId());
        } else {
            log.info("Calling Expedia reservation API");
            ItineraryReservationRequest request = createReservationRequest(context.getWorkflowEntity().getItinerary());
            try {
                ReservationResult reservation = wrap(() -> expediaClient.usingApi(itinerary.getApiVersion()).reserveItinerarySync(request,
                        itinerary.getExpediaReservationToken(),
                        itinerary.getCustomerIp(),
                        itinerary.getCustomerUserAgent(),
                        itinerary.getCustomerSessionId()));

                expediaItineraryId = reservation.getItineraryId();
                confirmationToken =
                        Helpers.retrieveConfirmationToken(reservation.getLinks().getResume().getHref(),
                                reservation.getItineraryId());
            } catch (ErrorException ex) {
                CancellationDetails reason = errorToCancellationDetails(ex.getError());
                Preconditions.checkNotNull(reason, "Request failed, but failure reason is unset");
                if (reason.getReason() == CancellationDetails.Reason.DUPLICATE) {
                    throw new UnexpectedItineraryDuplicateException(itinerary.getAffiliateId());
                }
                itinerary.setOrderCancellationDetails(reason);
                context.setState(EExpediaItemState.IS_CANCELLED);
                meters.incrementCancellationCounter(EPartnerId.PI_EXPEDIA, CancellationDetails.Reason.UNKNOWN);
                context.scheduleExternalEvent(context.getWorkflowEntity().getOrderWorkflowId(),
                        TServiceCancelled.newBuilder().setServiceId(context.getWorkflowEntity().getId().toString()).build());
                log.info("CANCELLED");
                return;
            }
        }
        itinerary.setExpediaItineraryId(expediaItineraryId);
        itinerary.setExpediaConfirmationToken(confirmationToken);
        Instant expiresAt =
                Instant.now().plus(expediaProperties.getReservation().getReservationDuration());
        itinerary.setExpiresAtInstant(expiresAt);
        item.setExpiresAt(expiresAt);
        fiscalItemCreator.addFiscalItems(item);
        item.setProviderId(expediaItineraryId);
        // TODO: may be set settlement items ?
        log.info("RESERVED");
        context.setState(EExpediaItemState.IS_RESERVED);
        context.scheduleExternalEvent(context.getWorkflowEntity().getOrderWorkflowId(),
                TServiceReserved.newBuilder().setServiceId(context.getWorkflowEntity().getId().toString()).build());
    }

    // suppress: first guest name is not nullable
    @SuppressWarnings("ConstantConditions")
    private ItineraryReservationRequest createReservationRequest(ExpediaHotelItinerary itinerary) {
        return ItineraryReservationRequest.create(
                itinerary.getAffiliateId(),
                Helpers.limitName(itinerary.getGuests().get(0).getFirstName()),
                Helpers.limitName(itinerary.getGuests().get(0).getLastName()),
                Helpers.convertEmailDomainNameToAscii(itinerary.getCustomerEmail()),
                itinerary.getCustomerPhone().replaceAll("[^\\d]", ""),
                expediaProperties.getReservation().getBilling().getAddressLine(),
                expediaProperties.getReservation().getBilling().getCity(),
                expediaProperties.getReservation().getBilling().getCountry());
    }
}
