package ru.yandex.travel.orders.workflows.orderitem.hotel;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Optional;

import javax.money.CurrencyUnit;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.hotels.common.orders.HotelItinerary;
import ru.yandex.travel.hotels.common.orders.MealData;
import ru.yandex.travel.hotels.common.orders.promo.AppliedPromoCampaigns;
import ru.yandex.travel.hotels.common.orders.promo.YandexPlusApplication;
import ru.yandex.travel.orders.entities.FiscalItem;
import ru.yandex.travel.orders.entities.FiscalItemType;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.VatType;
import ru.yandex.travel.orders.workflows.orderitem.FiscalItemCreator;

@Slf4j
public abstract class AbstractHotelFiscalItemCreator<T extends HotelOrderItem> implements FiscalItemCreator<T> {
    @Override
    public Collection<FiscalItem> createFiscalItems(T orderItem) {
        List<FiscalItem> items = new ArrayList<>(2);
        FiscalItem primaryItem = new FiscalItem();
        HotelItinerary itinerary = orderItem.getHotelItinerary();
        addProviderSpecificPropertiesToMainFiscalItem(primaryItem, orderItem);
        items.add(primaryItem);
        MealData mealData = itinerary.getMealData();
        Money primaryAmount = itinerary.getFiscalPrice();
        if (mealData != null) {
            Money totalMealAmount = mealData.getItems().stream().map(MealData.MealItem::getMealPrice)
                    .reduce(Money.zero(primaryAmount.getCurrency()), Money::add);
            if (totalMealAmount.isGreaterThanOrEqualTo(primaryAmount.add(Money.of(1, primaryAmount.getCurrency())))) {
                log.warn("Total meal amount ({}) exceeds stay amount ({}). Will NOT generate meal fiscal items",
                        totalMealAmount, primaryAmount);
            } else {
                for (var meal : mealData.getItems()) {
                    FiscalItem mealItem = new FiscalItem();
                    mealItem.setMoneyAmount(meal.getMealPrice());
                    mealItem.setType(FiscalItemType.HOTEL_MEAL);
                    mealItem.setVatType(VatType.fromEVat(meal.getMealVat()));
                    mealItem.setInn(primaryItem.getInn());          // same inn as primary stay
                    items.add(mealItem);
                    primaryAmount = primaryAmount.subtract(meal.getMealPrice());
                    addProviderSpecificPropertiesToMealFiscalItem(mealItem, orderItem);
                }
            }
        }
        primaryItem.setMoneyAmount(primaryAmount);
        addYandexPlusInformation(items, itinerary);
        return items;
    }

    private void addYandexPlusInformation(List<FiscalItem> fiscalItems, HotelItinerary itinerary) {
        int pointsToDistribute = Optional.ofNullable(itinerary.getAppliedPromoCampaigns())
                .map(AppliedPromoCampaigns::getYandexPlus)
                .filter(yandexPlusApplication -> yandexPlusApplication.getMode() == YandexPlusApplication.Mode.WITHDRAW)
                .map(YandexPlusApplication::getPoints).orElse(0);
        final CurrencyUnit currency = fiscalItems.get(0).getCurrency();
        Money totalToPay =
                fiscalItems.stream().map(FiscalItem::getMoneyAmount)
                        .reduce(Money.zero(currency), Money::add);

        log.info("Validating fiscal items: {} items, total-to-pay is {}, there are {} plus points to distribute",
                fiscalItems.size(),
                totalToPay,
                pointsToDistribute);
        if (pointsToDistribute != 0) {
            if (totalToPay.subtract(Money.of(fiscalItems.size(), currency)).isLessThan(Money.of(pointsToDistribute,
                    currency))) {
                log.warn("Not enough real money for all fiscal items and items with no real money are not allowed. " +
                        "Falling back to a single fiscal item");
                fiscalItems.removeIf(fi -> fi.getType() != FiscalItemType.EXPEDIA_HOTEL);
                Preconditions.checkState(fiscalItems.size() == 1, "Unexpected amount of remaining fiscal items");
                fiscalItems.get(0).setMoneyAmount(totalToPay);
            } else {
                log.info("Enough real money: will apply promo money to the items one by one");
            }
            for (var item : fiscalItems) {
                int maxPointsForItem = item.getMoneyAmount().getNumber().intValue() - 1;
                int points = Math.min(maxPointsForItem, pointsToDistribute);
                item.setYandexPlusToWithdraw(Money.of(points, item.getCurrency()));
                pointsToDistribute -= points;
                Preconditions.checkState(pointsToDistribute >= 0, "Negative points to distribute");
                if (pointsToDistribute == 0) {
                    break;
                }
            }
            Preconditions.checkState(pointsToDistribute == 0, "Undistributed plus-points left");
        }
        log.info("Final fiscal items: {}", fiscalItems);
    }

    protected abstract void addProviderSpecificPropertiesToMainFiscalItem(FiscalItem fiscalItem, T orderItem);

    protected void addProviderSpecificPropertiesToMealFiscalItem(FiscalItem fiscalItem, T orderItem) {
    }
}
