package ru.yandex.travel.orders.workflows.orderitem.jobs;

import java.time.Instant;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.protobuf.Message;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.repository.BNovoOrderItemRepository;
import ru.yandex.travel.orders.repository.BronevikOrderItemRepository;
import ru.yandex.travel.orders.repository.DolphinOrderItemRepository;
import ru.yandex.travel.orders.repository.ExpediaOrderItemRepository;
import ru.yandex.travel.orders.repository.OrderItemRepository;
import ru.yandex.travel.orders.repository.TravellineOrderItemRepository;
import ru.yandex.travel.orders.workflow.hotels.bnovo.proto.EBNovoItemState;
import ru.yandex.travel.orders.workflow.hotels.bronevik.proto.EBronevikItemState;
import ru.yandex.travel.orders.workflow.hotels.dolphin.proto.EDolphinItemState;
import ru.yandex.travel.orders.workflow.hotels.expedia.proto.EExpediaItemState;
import ru.yandex.travel.orders.workflow.hotels.travelline.proto.ETravellineItemState;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.WorkflowProcessService;

@Service
@Slf4j
@RequiredArgsConstructor
public class HotelOrderItemExpiredService {
    private static final UUID ZERO_UUID = UUID.fromString("00000000-0000-0000-0000-000000000000");
    private final BNovoOrderItemRepository bNovoOrderItemRepository;
    private final DolphinOrderItemRepository dolphinOrderItemRepository;
    private final ExpediaOrderItemRepository expediaOrderItemRepository;
    private final TravellineOrderItemRepository travellineOrderItemRepository;
    private final BronevikOrderItemRepository bronevikOrderItemRepository;
    private final OrderItemRepository orderItemRepository;
    private final WorkflowProcessService workflowProcessService;

    private final List<EBNovoItemState> bNovoItemStates = List.of(EBNovoItemState.IS_RESERVED);
    private final List<EDolphinItemState> dolphinItemStates = List.of(EDolphinItemState.IS_RESERVED);
    private final List<EExpediaItemState> expediaItemStates = List.of(EExpediaItemState.IS_RESERVED);
    private final List<ETravellineItemState> travellineItemStates = List.of(ETravellineItemState.IS_RESERVED);
    private final List<EBronevikItemState> bronevikItemStates = List.of(EBronevikItemState.IS_CHECKED_OFFER);

    @TransactionMandatory
    public List<UUID> fetchExpiredOrderItems(Set<UUID> active, int maxResultSize) {
        var activeIds = ensureSetIsNotEmpty(active);
        var bNovoItems = bNovoOrderItemRepository.getAllByIsExpiredIsFalseAndStateInAndExpiresAtIsBeforeAndWorkflowStateIsAndIdIsNotInOrderByExpiresAtAsc(
                bNovoItemStates,
                Instant.now(),
                EWorkflowState.WS_RUNNING,
                activeIds,
                PageRequest.of(0, maxResultSize));
        var dolphinItems = dolphinOrderItemRepository.getAllByIsExpiredIsFalseAndStateInAndExpiresAtIsBeforeAndWorkflowStateIsAndIdIsNotInOrderByExpiresAtAsc(
                dolphinItemStates,
                Instant.now(),
                EWorkflowState.WS_RUNNING,
                activeIds,
                PageRequest.of(0, maxResultSize));
        var expediaItems = expediaOrderItemRepository.getAllByIsExpiredIsFalseAndStateInAndExpiresAtIsBeforeAndWorkflowStateIsAndIdIsNotInOrderByExpiresAtAsc(
                expediaItemStates,
                Instant.now(),
                EWorkflowState.WS_RUNNING,
                activeIds,
                PageRequest.of(0, maxResultSize));
        var travellineItems = travellineOrderItemRepository.getAllByIsExpiredIsFalseAndStateInAndExpiresAtIsBeforeAndWorkflowStateIsAndIdIsNotInOrderByExpiresAtAsc(
                travellineItemStates,
                Instant.now(),
                EWorkflowState.WS_RUNNING,
                activeIds,
                PageRequest.of(0, maxResultSize));
        var bronevikItems = bronevikOrderItemRepository.getAllByIsExpiredIsFalseAndStateInAndExpiresAtIsBeforeAndWorkflowStateIsAndIdIsNotInOrderByExpiresAtAsc(
                bronevikItemStates,
                Instant.now(),
                EWorkflowState.WS_RUNNING,
                activeIds,
                PageRequest.of(0, maxResultSize));
        return Stream.of(bNovoItems, dolphinItems, expediaItems, travellineItems, bronevikItems)
                .flatMap(Collection::stream)
                .sorted(Comparator.comparing(OrderItem::getExpiresAt))
                .map(HotelOrderItem::getId)
                .collect(Collectors.toList());
    }

    @TransactionMandatory
    public long countExpiredOrderItems(Set<UUID> active) {
        var activeIds = ensureSetIsNotEmpty(active);
        var bNovo = bNovoOrderItemRepository.countAllByIsExpiredIsFalseAndStateInAndExpiresAtIsBeforeAndWorkflowStateIsAndIdIsNotIn(
                bNovoItemStates,
                Instant.now(),
                EWorkflowState.WS_RUNNING,
                activeIds);
        var dolphin = dolphinOrderItemRepository.countAllByIsExpiredIsFalseAndStateInAndExpiresAtIsBeforeAndWorkflowStateIsAndIdIsNotIn(
                dolphinItemStates,
                Instant.now(),
                EWorkflowState.WS_RUNNING,
                activeIds);
        var expedia = expediaOrderItemRepository.countAllByIsExpiredIsFalseAndStateInAndExpiresAtIsBeforeAndWorkflowStateIsAndIdIsNotIn(
                expediaItemStates,
                Instant.now(),
                EWorkflowState.WS_RUNNING,
                activeIds);
        var travelline = travellineOrderItemRepository.countAllByIsExpiredIsFalseAndStateInAndExpiresAtIsBeforeAndWorkflowStateIsAndIdIsNotIn(
                travellineItemStates,
                Instant.now(),
                EWorkflowState.WS_RUNNING,
                activeIds);
        var bronevik = bronevikOrderItemRepository.countAllByIsExpiredIsFalseAndStateInAndExpiresAtIsBeforeAndWorkflowStateIsAndIdIsNotIn(
                bronevikItemStates,
                Instant.now(),
                EWorkflowState.WS_RUNNING,
                activeIds);
        return bNovo + dolphin + expedia + travelline + bronevik;
    }

    @TransactionMandatory
    public void startCancellation(UUID orderItemId) {
        OrderItem orderItem = orderItemRepository.getOne(orderItemId);
        try (var ignored = NestedMdc.forEntity(orderItem.getId(), orderItem.getLogEntityType())) {
            if (orderItem.isExpired()) {
                log.info("Already expired, returning");
                return;
            }

            log.info("OrderItem {} expired at {}", orderItemId, orderItem.getExpiresAt());

            orderItem.setExpired(true);

            Message reservationExpired;
            switch (orderItem.getPublicType()) {
                case PT_BNOVO_HOTEL:
                    reservationExpired = ru.yandex.travel.orders.workflow.hotels.bnovo.proto.TReservationExpired
                            .newBuilder()
                            .build();
                    break;
                case PT_DOLPHIN_HOTEL:
                    reservationExpired = ru.yandex.travel.orders.workflow.hotels.dolphin.proto.TReservationExpired
                            .newBuilder()
                            .build();
                    break;
                case PT_EXPEDIA_HOTEL:
                    reservationExpired = ru.yandex.travel.orders.workflow.hotels.expedia.proto.TReservationExpired
                            .newBuilder()
                            .build();
                    break;
                case PT_TRAVELLINE_HOTEL:
                    reservationExpired = ru.yandex.travel.orders.workflow.hotels.travelline.proto.TReservationExpired
                            .newBuilder()
                            .build();
                    break;
                case PT_BRONEVIK_HOTEL:
                    reservationExpired = ru.yandex.travel.orders.workflow.hotels.bronevik.proto.TReservationExpired
                            .newBuilder()
                            .build();
                    break;
                default:
                    throw new IllegalArgumentException(orderItem.getPublicType() + " order item not expected in hotelExpirationService");
            }
            workflowProcessService.scheduleEvent(orderItem.getWorkflow().getId(), reservationExpired);
        }
    }

    private Set<UUID> ensureSetIsNotEmpty(Set<UUID> uuidSet) {
        Set<UUID> notEmptySet = uuidSet;
        if (notEmptySet == null || notEmptySet.size() == 0) {
            notEmptySet = new HashSet<>();
            notEmptySet.add(ZERO_UUID);
        }
        return notEmptySet;
    }
}
