package ru.yandex.travel.orders.workflows.orderitem.jobs;

import java.util.Collection;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.support.DefaultTransactionDefinition;

import ru.yandex.travel.orders.workflows.order.hotel.HotelWorkflowProperties;
import ru.yandex.travel.spring.tx.ForcedRollbackTxManagerWrapper;
import ru.yandex.travel.task_processor.AbstractTaskKeyProvider;
import ru.yandex.travel.task_processor.TaskKeyProvider;
import ru.yandex.travel.task_processor.TaskProcessor;
import ru.yandex.travel.task_processor.TaskProcessorProperties;

@Configuration
@RequiredArgsConstructor
@EnableConfigurationProperties(HotelWorkflowProperties.class)
public class HotelOrderItemTaskProcessorConfiguration {
    private final HotelOrderItemExpiredService hotelOrderItemExpiredService;
    private final ForcedRollbackTxManagerWrapper forcedRollbackTxManagerWrapper;
    private final HotelWorkflowProperties hotelWorkflowProperties;

    @Bean
    @ConditionalOnProperty(value = "hotel-workflow.check-expiration-task.enabled")
    public TaskProcessor<UUID> hotelOrderItemCheckingExpirationProcessor() {

        TaskKeyProvider<UUID> taskKeyProvider = new AbstractTaskKeyProvider<>() {
            @Override
            public Collection<UUID> getPendingTaskKeys(int maxResultSize) {
                return hotelOrderItemExpiredService.fetchExpiredOrderItems(getLockedTaskKeys(),
                        maxResultSize);
            }

            @Override
            public long getPendingTasksCount() {
                return hotelOrderItemExpiredService.countExpiredOrderItems(getLockedTaskKeys());
            }
        };
        DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
        transactionDefinition.setName("HotelItemsCheckingExpiration");
        transactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
        TransactionDefinition txDefinition = new DefaultTransactionDefinition(transactionDefinition);

        return new TaskProcessor<>(taskKeyProvider, hotelOrderItemExpiredService::startCancellation,
                forcedRollbackTxManagerWrapper, txDefinition,
                TaskProcessorProperties.builder().name("HotelItemsCheckingExpiration")
                        .daemonPoolThreads(true).gracefulPoolShutdown(true)
                        .poolSize(hotelWorkflowProperties.getCheckExpirationTask().getNumberOfItems())
                        .initialStartDelay(hotelWorkflowProperties.getCheckExpirationTask().getInitialStartDelay())
                        .scheduleRate(hotelWorkflowProperties.getCheckExpirationTask().getPeriod())
                        .build());
    }
}
