package ru.yandex.travel.orders.workflows.orderitem.suburban.handlers;

import java.time.Instant;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.SuburbanOrderItem;
import ru.yandex.travel.orders.services.suburban.environment.SuburbanOrderItemEnvProvider;
import ru.yandex.travel.orders.services.suburban.providers.SuburbanProviderBase;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.orderitem.suburban.proto.TConfirmationCommit;
import ru.yandex.travel.orders.workflow.orderitem.suburban.proto.TGetTicketBarcode;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TReservationExpired;
import ru.yandex.travel.suburban.exceptions.SuburbanRetryableException;
import ru.yandex.travel.suburban.model.SuburbanReservation;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;
import ru.yandex.travel.workflow.exceptions.RetryableException;


@Slf4j
@RequiredArgsConstructor
@IgnoreEvents(types = TReservationExpired.class)
public class ConfirmingStateHandler extends BaseHandler {
    private final SuburbanOrderItemEnvProvider suburbanEnvProvider;

    @HandleEvent
    public void handleConfirmationCommit(TConfirmationCommit event,
                                         StateContext<EOrderItemState, SuburbanOrderItem> ctx) {
        SuburbanOrderItem orderItem = ctx.getWorkflowEntity();
        SuburbanProviderBase provider = suburbanEnvProvider.createEnv(orderItem).createSuburbanProvider();
        SuburbanReservation.WorkflowData workflowData = orderItem.getPayload().getWorkflowData();

        SuburbanProviderBase.ConfirmResult confirmResult;
        try {
            confirmResult = provider.confirmOrder();
        } catch (SuburbanRetryableException ex) {
            var attemptsCount = ctx.getAttempt() + 1;
            Instant startedAt = workflowData.getConfirmStartedAt();
            if ((attemptsCount < provider.getConfirmMaxAttempts()) &&
                    (startedAt == null || Instant.now().isBefore(startedAt.plus(provider.getConfirmRetryingTime())))){
                throw new RetryableException(ex);
            }
            else {
                String msg = String.format("Suburban confirm %s retries failed: %s", attemptsCount, ex);
                log.error(msg, ex);
                setItemCancelling(ctx, msg);
                return;
            }
        } catch (Exception ex) {
            String msg = String.format("Suburban confirm failed: %s", ex);
            log.error(msg, ex);
            setItemCancelling(ctx, msg);
            return;
        }

        log.info("Suburban confirm done. Ticket number: {}", confirmResult.getTicketNumber());

        if (provider.getGetTicketBarcodeMaxAttempts() > 0) {
            workflowData.setGetTicketBarcodeStartedAt(Instant.now());
            ctx.scheduleEvent(TGetTicketBarcode.getDefaultInstance());
        } else {
            orderItem.setConfirmedAt(Instant.now());
            provider.onConfirm();
            setItemConfirmed(ctx);
            setClearTime(provider, orderItem);
        }
    }

    @HandleEvent
    public void handleGetTicketBarcode(TGetTicketBarcode event,
                                       StateContext<EOrderItemState, SuburbanOrderItem> ctx) {
        // Отдельное получение штрих-кода запускается только для ИМ
        SuburbanOrderItem orderItem = ctx.getWorkflowEntity();
        SuburbanProviderBase provider = suburbanEnvProvider.createEnv(orderItem).createSuburbanProvider();
        SuburbanReservation.WorkflowData workflowData = orderItem.getPayload().getWorkflowData();

        try {
            provider.getTicketBarcode();
        } catch (SuburbanRetryableException ex) {
            var attemptsCount = ctx.getAttempt() + 1;
            Instant startedAt = workflowData.getConfirmStartedAt();
            if ((attemptsCount < provider.getGetTicketBarcodeMaxAttempts()) && (startedAt == null
                    || Instant.now().isBefore(startedAt.plus(provider.getGetTicketBarcodeRetryingTime())))) {
                throw new RetryableException(ex);
            } else {
                String msg = String.format("Suburban getTicketBarcode %s retries failed: %s", attemptsCount, ex);
                log.error(msg, ex);
                setItemCancelling(ctx, msg);
                return;
            }
        } catch (Exception ex) {
            String msg = String.format("Suburban getTicketBarcode failed: %s", ex);
            log.error(msg, ex);
            setItemCancelling(ctx, msg);
            return;
        }

        log.info("Ticket barcode received");
        orderItem.setConfirmedAt(Instant.now());
        provider.onConfirm();
        setItemConfirmed(ctx);
        setClearTime(provider, orderItem);
    }

    private void setClearTime(SuburbanProviderBase provider, SuburbanOrderItem orderItem) {
        // Если нужно, то задаем время очистки платежа, делается, если нужно очистить платеж сразу, а не ждать днями
        if (provider.getPaymentClearingDelay() != null) {
            Invoice invoice = orderItem.getOrder().getInvoices().get(0);
            Instant clearTime = Instant.now().plus(provider.getPaymentClearingDelay());
            invoice.setClearAt(clearTime);
        }
    }
}
