package ru.yandex.travel.orders.workflows.orderitem.suburban.handlers;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.orders.entities.FiscalItem;
import ru.yandex.travel.orders.entities.SuburbanOrderItem;
import ru.yandex.travel.orders.entities.VatType;
import ru.yandex.travel.orders.services.suburban.environment.SuburbanOrderItemEnvProvider;
import ru.yandex.travel.orders.services.suburban.providers.SuburbanProviderBase;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.orderitem.suburban.proto.TBeforeReservation;
import ru.yandex.travel.orders.workflow.orderitem.suburban.proto.TReservationCommit;
import ru.yandex.travel.suburban.exceptions.SuburbanRetryableException;
import ru.yandex.travel.suburban.model.SuburbanReservation;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.exceptions.RetryableException;


@Slf4j
@RequiredArgsConstructor
public class ReservingStateHandler extends BaseHandler {
    private final SuburbanOrderItemEnvProvider suburbanEnvProvider;

    @HandleEvent
    public void handleBeforeReservation(TBeforeReservation event,
                                        StateContext<EOrderItemState, SuburbanOrderItem> ctx) {
        // Событие ловится только для того, чтобы записать факт срабатывания в Solomon
        log.info("TBeforeReservation event received");
    }

    @HandleEvent
    public void handleReservationCommit(TReservationCommit event,
                                        StateContext<EOrderItemState, SuburbanOrderItem> ctx) {

        SuburbanOrderItem orderItem = ctx.getWorkflowEntity();
        SuburbanProviderBase provider = suburbanEnvProvider.createEnv(orderItem).createSuburbanProvider();

        SuburbanProviderBase.BookResult bookResult;
        try {
            bookResult = provider.bookOrder();
        } catch (SuburbanRetryableException ex) {
            var attemptsCount = ctx.getAttempt() + 1;
            if (attemptsCount < provider.getBookMaxAttempts()) {
                throw new RetryableException(ex);
            } else {
                String msg = String.format("Suburban book %s retries failed: %s", attemptsCount, ex);
                log.error(msg, ex);
                setItemCancelling(ctx, msg);
                return;
            }
        } catch (Exception ex) {
            String msg = String.format("Suburban book failed: %s", ex);
            log.error(msg, ex);
            setItemCancelling(ctx, msg);
            return;
        }

        SuburbanReservation reservation = orderItem.getPayload();

        // https://st.yandex-team.ru/RASPFRONT-9234
        Money actualPrice = bookResult.getPrice();
        if (actualPrice.isGreaterThan(reservation.getPrice())) {
            setItemCancelling(ctx, String.format(
                    "Suburban book: actual book price is greater than known price: %s > %s",
                    actualPrice, reservation.getPrice()));
            return;
        }

        log.info("Suburban book done. Provider order: {}", reservation.getProviderOrderId());
        reservation.setPrice(actualPrice);
        orderItem.setExpiresAt(bookResult.getExpiresAt());
        generateFiscalItems(orderItem, provider);
        setItemReserved(ctx);
    }

    public void generateFiscalItems(SuburbanOrderItem orderItem, SuburbanProviderBase provider) {
        SuburbanReservation reservation = orderItem.getPayload();
        orderItem.addFiscalItem(FiscalItem.builder()
                .moneyAmount(reservation.getPrice())
                .vatType(VatType.VAT_NONE)
                .type(provider.getFiscalItemType())
                .title(provider.getFiscalTitle())
                .inn(provider.getFiscalInn()).build()
        );
    }
}
