package ru.yandex.travel.orders.workflows.orderitem.suburban.jobs;

import java.time.Instant;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.orders.entities.SuburbanOrderItem;
import ru.yandex.travel.orders.repository.SuburbanOrderItemRepository;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.orderitem.suburban.proto.TReservationExpired;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.WorkflowMessageSender;

@Service
@Slf4j
@RequiredArgsConstructor
public class SuburbanOrderItemExpiredService {
    private final SuburbanOrderItemRepository suburbanOrderItemRepository;
    private final WorkflowMessageSender workflowMessageSender;

    private final List<EOrderItemState> orderItemStates = List.of(EOrderItemState.IS_RESERVED);

    @TransactionMandatory
    public List<UUID> fetchExpiredOrderItems(Set<UUID> active, int maxResultSize) {
        return suburbanOrderItemRepository.getExpiredOrderItems(
                Instant.now(),
                orderItemStates,
                EWorkflowState.WS_RUNNING,
                active.isEmpty() ? SuburbanOrderItemRepository.EMPTY_EXCLUDE_UIDS : active,
                PageRequest.of(0, maxResultSize)
        );
    }

    @TransactionMandatory
    public long countExpiredOrderItems(Set<UUID> active) {
        return suburbanOrderItemRepository.countExpiredOrderItems(
                Instant.now(),
                orderItemStates,
                EWorkflowState.WS_RUNNING,
                active.isEmpty() ? SuburbanOrderItemRepository.EMPTY_EXCLUDE_UIDS : active);
    }

    @TransactionMandatory
    public void startCancellation(UUID orderItemId) {
        SuburbanOrderItem suburbanOrderItem = suburbanOrderItemRepository.getOne(orderItemId);
        try (var ignored = NestedMdc.forEntity(suburbanOrderItem.getId(), suburbanOrderItem.getEntityType())) {
            if (suburbanOrderItem.isExpired()) {
                log.info("Already expired, returning");
                return;
            }

            log.info("OrderItem {} expired at {}", orderItemId, suburbanOrderItem.getExpiresAt());

            suburbanOrderItem.setExpired(true);
            workflowMessageSender.scheduleEvent(suburbanOrderItem.getWorkflow().getId(),
                    TReservationExpired.getDefaultInstance());
        }
    }
}
