package ru.yandex.travel.orders.workflows.orderitem.train;

import io.opentracing.Tracer;
import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.core.env.Environment;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.support.DefaultTransactionDefinition;
import org.springframework.transaction.support.TransactionTemplate;
import org.springframework.validation.annotation.Validated;

import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.orders.commons.proto.TTrainTestContext;
import ru.yandex.travel.orders.repository.mock.MockTrainOrderItemRepository;
import ru.yandex.travel.orders.repository.mock.MockTrainOrderRepository;
import ru.yandex.travel.orders.services.mock.MockImClient;
import ru.yandex.travel.orders.services.mock.MockImResponseGenerator;
import ru.yandex.travel.orders.services.train.ImClientProvider;
import ru.yandex.travel.train.partners.im.CustomerInfoRepository;
import ru.yandex.travel.train.partners.im.ImClient;
import ru.yandex.travel.train.partners.im.ImClientConfigurationHelper;
import ru.yandex.travel.train.partners.im.ImProperties;
import ru.yandex.travel.workflow.single_operation.SingleOperationService;

@Configuration
@RequiredArgsConstructor
@EnableConfigurationProperties({ImClientOrchConfiguration.ImPropertiesImpl.class, MockImClientProperties.class})
public class ImClientOrchConfiguration {
    private final ImClientOrchConfiguration.ImPropertiesImpl imProperties;

    private final Environment environment;

    @Bean
    public AsyncHttpClientWrapper imAhcClient(Tracer tracer) {
        return ImClientConfigurationHelper.createImAhcClient(tracer);
    }

    @Bean
    @Primary
    public ImClient defaultImClient(AsyncHttpClientWrapper imAhcClient, CustomerInfoRepository customerInfoRepository) {
        return ImClientConfigurationHelper.createImClient(imProperties, imAhcClient, customerInfoRepository);
    }

    @Bean
    @ConditionalOnProperty(value = "mock-im-client.enabled")
    public MockImClient mockImClient(MockTrainOrderRepository mockTrainOrderRepository,
                                     MockTrainOrderItemRepository mockTrainOrderItemRepository,
                                     PlatformTransactionManager transactionManager,
                                     SingleOperationService singleOperationService,
                                     MockImResponseGenerator mockImResponseGenerator) {
        if (environment.acceptsProfiles("prod")) {
            throw new RuntimeException("\n\nATTENTION: MOCK IM CLIENT MUST BE DISABLED FOR PROD ENVIRONMENT\n\n");
        }
        DefaultTransactionDefinition defaultTransactionDefinition = new DefaultTransactionDefinition();
        defaultTransactionDefinition.setName("MockImClient");
        defaultTransactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        defaultTransactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
        TransactionDefinition transactionDefinition = new DefaultTransactionDefinition(defaultTransactionDefinition);
        TransactionTemplate transactionTemplate = new TransactionTemplate(transactionManager, transactionDefinition);
        return new MockImClient(mockTrainOrderRepository, mockTrainOrderItemRepository, transactionTemplate,
                singleOperationService, mockImResponseGenerator);
    }

    @Bean
    public ImClientProvider imClientProvider(ImClient defaultImClient,
                                             @Autowired(required = false) MockImClient mockImClient) {
        return orderItem -> {
            if (orderItem != null) {
                var testContext = orderItem.getTestContext();
                if (testContext instanceof TTrainTestContext) {
                    return mockImClient;
                } else {
                    return defaultImClient;
                }
            } else {
                return defaultImClient;
            }
        };
    }

    @Validated
    @ConfigurationProperties(value = "im", ignoreUnknownFields = false)
    public static class ImPropertiesImpl extends ImProperties {
    }
}
