package ru.yandex.travel.orders.workflows.orderitem.train;

import java.text.MessageFormat;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.Map;

import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;

import ru.yandex.travel.orders.entities.VatType;
import ru.yandex.travel.train.model.ErrorCode;
import ru.yandex.travel.train.model.ErrorInfo;
import ru.yandex.travel.train.partners.im.ImClientBonusCardException;
import ru.yandex.travel.train.partners.im.ImClientException;
import ru.yandex.travel.train.partners.im.ImClientIOException;
import ru.yandex.travel.train.partners.im.ImClientInvalidDocumentNumberException;
import ru.yandex.travel.train.partners.im.ImClientInvalidPassengerDataException;
import ru.yandex.travel.train.partners.im.ImClientNameRequiredLatinLettersException;
import ru.yandex.travel.train.partners.im.ImClientNoPlacesException;
import ru.yandex.travel.train.partners.im.ImClientRetryableException;
import ru.yandex.travel.train.partners.im.ImClientTariffErrorException;
import ru.yandex.travel.train.partners.im.ImClientTooLateException;
import ru.yandex.travel.train.partners.im.ImClientTooLateForOrderException;
import ru.yandex.travel.train.partners.im.ImClientUnknownPartnerErrorException;

public class ImHelpers {
    public static final String MSK_TZ = "Europe/Moscow";

    public static Instant fromLocalDateTime(LocalDateTime dateTime, String zone) {
        if (dateTime == null) {
            return null;
        }
        ZoneId zoneId = ZoneId.of(zone);
        return dateTime.atZone(zoneId).toInstant();
    }

    public static LocalDateTime toLocalDateTime(Instant dateTime, String zone) {
        if (dateTime == null) {
            return null;
        }
        ZoneId zoneId = ZoneId.of(zone);
        return dateTime.atZone(zoneId).toLocalDateTime();
    }

    private static final Map<Double, VatType> rateToVatType = new ImmutableMap.Builder<Double, VatType>()
            .put(0.0, VatType.VAT_0)
            .put(9.09, VatType.VAT_10_110)
            .put(10.0, VatType.VAT_10)
            .put(15.25, VatType.VAT_18_118)
            .put(18.0, VatType.VAT_18)
            .put(16.67, VatType.VAT_20_120)
            .put(20.0, VatType.VAT_20)
            .build();

    public static VatType vatFromRate(Double rate) {
        if (rate == null) {
            return VatType.VAT_NONE;
        }
        VatType result = rateToVatType.get(rate);
        if (result == null) {
            throw new IllegalArgumentException(MessageFormat.format("Unknown fiscal rate {0}", rate));
        }
        return result;
    }

    public static ErrorInfo createErrorInfo(Throwable e) {
        var errorInfo = new ErrorInfo();
        if (e instanceof ImClientIOException) {
            errorInfo.setCode(ErrorCode.UNKNOWN_PARTNER_ERROR);
        } else if (e instanceof ImClientRetryableException) {
            errorInfo.setCode(ErrorCode.TRY_LATER);
        } else if (e instanceof ImClientNoPlacesException) {
            errorInfo.setCode(ErrorCode.NO_PLACES);
        } else if (e instanceof ImClientBonusCardException) {
            errorInfo.setCode(ErrorCode.INVALID_BONUS_CARD);
            errorInfo.setMessageParams(((ImClientBonusCardException) e).getMessageParams());
            errorInfo.setMessage(e.getMessage());
        } else if (e instanceof ImClientNameRequiredLatinLettersException) {
            errorInfo.setCode(ErrorCode.NAME_REQUIRED_LATIN_LETTERS);
        } else if (e instanceof ImClientInvalidDocumentNumberException) {
            errorInfo.setCode(ErrorCode.INVALID_DOCUMENT_NUMBER);
        } else if (e instanceof ImClientTariffErrorException) {
            errorInfo.setCode(ErrorCode.TARIFF_ERROR);
        } else if (e instanceof ImClientInvalidPassengerDataException) {
            errorInfo.setCode(ErrorCode.UNKNOWN_PARTNER_ERROR);
        } else if (e instanceof ImClientTooLateForOrderException) {
            errorInfo.setCode(ErrorCode.TOO_LATE_FOR_ORDER);
        } else if (e instanceof ImClientTooLateException) {
            errorInfo.setCode(ErrorCode.TOO_LATE);
        } else if (e instanceof ImClientUnknownPartnerErrorException) {
            errorInfo.setCode(ErrorCode.UNKNOWN_PARTNER_ERROR);
        } else if (e instanceof ImClientException) {
            errorInfo.setMessage(e.getMessage());
            errorInfo.setMessageParams(((ImClientException) e).getMessageParams());
        }
        return errorInfo;
    }

    /**
     * <b>Валидация ИМ, если номер не контактный:</b><p>
     *
     * Формат телефона:
     * +7 и еще строго 10 цифр;
     * +1 и еще строго 10 цифр;
     * + и еще не менее 8 и не более 14 цифр, при этом первыми не могут быть цифры: 0,
     * 1, 7, 89.
     * при оформлении дорожных карт телефон должен начинаться с цифры 8, далее 10
     * цифр, либо со знака «+» далее не более 14 цифр.
     * Знак плюс перед телефоном обязателен!
     * Все российские телефоны записываются в формате +7хххххххххх.
     *
     * @return a well formatted version of the given phone number
     *         or null if it doesn't match the IM phone formatting rules
     */
    public static String formatImNonContactPhone(String phone, boolean isForeignDocument) {
        if (Strings.isNullOrEmpty(phone)) {
            return null;
        }
        String normalized = phone.replaceAll("[^\\d]+", "");
        if (normalized.startsWith("007")) {
            normalized = "7" + normalized.substring("007".length());
        }
        if (isForeignDocument) {
            normalized = "+" + normalized;
            if (normalized.length() >= 6 && normalized.length() <= 30) {
                return normalized;
            }
        } else {
            if (normalized.startsWith("8") && normalized.length() == 11) {
                return "+7" + normalized.substring(1);
            }
            if (normalized.startsWith("7") && normalized.length() >= 5 && normalized.length() <= 11) {
                return "+" + normalized;
            }
            if (normalized.length() == 10) {
                return "+7" + normalized;
            }
        }
        return null;
    }
}
