package ru.yandex.travel.orders.workflows.orderitem.train;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import lombok.experimental.UtilityClass;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.streams.CustomCollectors;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.entities.FiscalItem;
import ru.yandex.travel.orders.entities.InvoiceItem;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.entities.TrainTicketRefund;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.train.model.TrainPassenger;
import ru.yandex.travel.train.model.TrainTicket;
import ru.yandex.travel.train.model.TrainTicketRefundStatus;
import ru.yandex.travel.train.model.refund.InsuranceItemInfo;
import ru.yandex.travel.train.model.refund.PassengerRefundInfo;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderInfoResponse;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderItemResponse;

@UtilityClass
public class TrainOrderItemHelpers {
    public static void setTrainTicketRefundStatusForBlanks(TrainOrderItem orderItem, Set<Integer> blankIds,
                                                           TrainTicketRefundStatus status) {
        for (TrainPassenger passenger : getPassengersByBlankIds(orderItem, blankIds)) {
            TrainTicket ticket = passenger.getTicket();
            ticket.setRefundStatus(status);
        }
    }

    public static List<TrainPassenger> getPassengersByBlankIds(TrainOrderItem orderItem, Set<Integer> blankIds) {
        return orderItem.getReservation().getPassengers().stream()
                .filter(p -> blankIds.contains(p.getTicket().getBlankId()))
                .collect(Collectors.toList());
    }

    public static Map<Long, Money> createRefundFiscalItems(TrainTicketRefund refund, TrainOrderItem orderItem) {
        Map<Integer, Long> fiscalItemIdByInternalId = orderItem.getFiscalItems().stream()
                .collect(Collectors.toMap(FiscalItem::getInternalId, FiscalItem::getId));
        Map<Integer, TrainPassenger> passengersByCustomerId = orderItem.getPayload().getPassengers().stream()
                .collect(Collectors.toMap(TrainPassenger::getCustomerId, x -> x));
        Map<Long, Money> sourceFiscalItems = getPaidAmounts(orderItem.getOrder());
        Map<Long, Money> targetFiscalItems = new HashMap<>();
        for (PassengerRefundInfo passengerRefundInfo : refund.getPayload().getRefundedItems()) {
            TrainPassenger passenger = passengersByCustomerId.get(passengerRefundInfo.getCustomerId());

            var serviceFiscalItemId =
                    fiscalItemIdByInternalId.get(passenger.getTicket().getServiceFiscalItemInternalId());
            var tariffFiscalItemId =
                    fiscalItemIdByInternalId.get(passenger.getTicket().getTariffFiscalItemInternalId());
            var feeFiscalItemId =
                    fiscalItemIdByInternalId.get(passenger.getTicket().getFeeFiscalItemInternalId());

            var zeroAmount = Money.zero(orderItem.getOrder().getCurrency());
            var serviceSourceAmount = sourceFiscalItems.getOrDefault(serviceFiscalItemId, zeroAmount);
            var tariffSourceAmount = sourceFiscalItems.getOrDefault(tariffFiscalItemId, zeroAmount);
            var feeSourceAmount = sourceFiscalItems.getOrDefault(feeFiscalItemId, zeroAmount);

            var serviceRefundAmount = serviceSourceAmount;
            var tariffRefundAmount = passengerRefundInfo.getActualRefundTicketAmount();
            var feeRefundAmount = passengerRefundInfo.getCalculatedRefundFeeAmount();
            var insuranceRefundAmount = passengerRefundInfo.getCalculatedRefundInsuranceAmount();

            if (tariffRefundAmount.isLessThan(serviceRefundAmount)) {
                serviceRefundAmount = tariffRefundAmount;
            }
            tariffRefundAmount = tariffRefundAmount.subtract(serviceRefundAmount);

            if (!serviceRefundAmount.isZero()) {
                targetFiscalItems.put(serviceFiscalItemId, serviceSourceAmount.subtract(serviceRefundAmount));
            }
            if (!tariffRefundAmount.isZero()) {
                targetFiscalItems.put(tariffFiscalItemId, tariffSourceAmount.subtract(tariffRefundAmount));
            }
            if (feeRefundAmount != null && !feeRefundAmount.isZero()) {
                targetFiscalItems.put(feeFiscalItemId, feeSourceAmount.subtract(feeRefundAmount));
            }
            if (insuranceRefundAmount != null && !insuranceRefundAmount.isZero()) {
                var insuranceFiscalItemId =
                        fiscalItemIdByInternalId.get(passenger.getInsurance().getFiscalItemInternalId());
                var insuranceSourceAmount = sourceFiscalItems.getOrDefault(insuranceFiscalItemId, zeroAmount);
                targetFiscalItems.put(insuranceFiscalItemId, insuranceSourceAmount.subtract(insuranceRefundAmount));
            }
        }
        return targetFiscalItems;
    }

    @SuppressWarnings("SuspiciousMethodCalls")
    private static Map<Long, Money> getPaidAmounts(Order order) {
        return Optional.ofNullable(order.getCurrentInvoice()).stream()
                .filter(invoice -> !TrustInvoice.UNPAID_INVOICE_STATES.contains(invoice.getInvoiceState()))
                .flatMap(invoice -> invoice.getInvoiceItems().stream())
                .collect(Collectors.groupingBy(
                        InvoiceItem::getFiscalItemId,
                        CustomCollectors.summingMoney(InvoiceItem::getPriceMoney, order.getCurrency())
                ));
    }

    public static List<TrainOrderItem> getSlaves(TrainOrderItem orderItem) {
        if (orderItem.getPayload().isMasterItem()) {
            return orderItem.getOrder().getOrderItems().stream()
                    .filter(x -> x.getPublicType() == EServiceType.PT_TRAIN)
                    .map(x -> (TrainOrderItem) x)
                    .filter(x -> x.getPayload().isSlaveItem())
                    .collect(Collectors.toList());
        }
        return List.of();
    }

    public static OrderItemResponse findInsuranceRefundItem(OrderInfoResponse orderInfoResponse,
                                                            InsuranceItemInfo item) {
        return orderInfoResponse.findRefundInsuranceItems().stream()
                .filter(x -> item.getRefundReferenceId().equals(x.getAgentReferenceId()))
                .findFirst().orElse(null);
    }
}
