package ru.yandex.travel.orders.workflows.orderitem.train;

import com.google.common.collect.ImmutableMap;
import com.netflix.concurrency.limits.limiter.SimpleLimiter;
import lombok.RequiredArgsConstructor;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.management.StarTrekService;
import ru.yandex.travel.orders.repository.OrderRefundRepository;
import ru.yandex.travel.orders.repository.TrainInsuranceRefundRepository;
import ru.yandex.travel.orders.repository.TrainOrderItemRepository;
import ru.yandex.travel.orders.repository.TrainTicketRefundRepository;
import ru.yandex.travel.orders.services.AuthorizationService;
import ru.yandex.travel.orders.services.DeduplicationService;
import ru.yandex.travel.orders.services.finances.providers.TrainFinancialDataProvider;
import ru.yandex.travel.orders.services.partners.BillingPartnerService;
import ru.yandex.travel.orders.services.train.FeeCalculationService;
import ru.yandex.travel.orders.services.train.ImClientProvider;
import ru.yandex.travel.orders.services.train.RebookingService;
import ru.yandex.travel.orders.services.train.TrainDiscountService;
import ru.yandex.travel.orders.services.train.TrainMeters;
import ru.yandex.travel.orders.services.train.tariffinfo.TrainTariffInfoDataProvider;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflows.orderitem.train.handlers.CalculatingFeeStateHandler;
import ru.yandex.travel.orders.workflows.orderitem.train.handlers.CancelledStateHandler;
import ru.yandex.travel.orders.workflows.orderitem.train.handlers.CancellingInsuranceStateHandler;
import ru.yandex.travel.orders.workflows.orderitem.train.handlers.CancellingStateHandler;
import ru.yandex.travel.orders.workflows.orderitem.train.handlers.CheckingConfirmationStateHandler;
import ru.yandex.travel.orders.workflows.orderitem.train.handlers.ConfirmedStateHandler;
import ru.yandex.travel.orders.workflows.orderitem.train.handlers.ConfirmingStateHandler;
import ru.yandex.travel.orders.workflows.orderitem.train.handlers.InsurancePricingStateHandler;
import ru.yandex.travel.orders.workflows.orderitem.train.handlers.NewStateHandler;
import ru.yandex.travel.orders.workflows.orderitem.train.handlers.RefundedStateHandler;
import ru.yandex.travel.orders.workflows.orderitem.train.handlers.RefundingTicketStateHandler;
import ru.yandex.travel.orders.workflows.orderitem.train.handlers.ReservedStateHandler;
import ru.yandex.travel.orders.workflows.orderitem.train.handlers.ReservingInsuranceStateHandler;
import ru.yandex.travel.orders.workflows.orderitem.train.handlers.ReservingStateHandler;
import ru.yandex.travel.workflow.StatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.WorkflowEventHandler;
import ru.yandex.travel.workflow.base.ProxyStateMapWorkflowEventHandler;
import ru.yandex.travel.workflow.repository.EntityStateTransitionRepository;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

@Configuration
@RequiredArgsConstructor
@SuppressWarnings("SameParameterValue")
public class TrainWorkflowConfiguration {
    private final AuthorizationService authorizationService;
    private final DeduplicationService deduplicationService;
    private final EntityStateTransitionRepository entityStateTransitionRepository;
    private final FeeCalculationService feeCalculationService;
    private final ImClientProvider imClientProvider;
    private final OrderRefundRepository orderRefundRepository;
    private final StarTrekService starTrekService;
    private final TrainDiscountService trainDiscountService;
    private final TrainInsuranceRefundRepository trainInsuranceRefundRepository;
    private final TrainTariffInfoDataProvider trainTariffInfoDataProvider;
    private final TrainTicketRefundRepository trainTicketRefundRepository;
    private final TrainWorkflowProperties trainWorkflowProperties;
    private final WorkflowRepository workflowRepository;
    private final BillingPartnerService billingPartnerService;
    private final RebookingService rebookingService;
    private final TrainOrderItemRepository trainOrderItemRepository;
    private final TrainMeters trainMeters;
    private final SimpleLimiter<Void> reserveCallLimiter;
    private final TrainFinancialDataProvider trainFinancialDataProvider;

    @Bean
    public WorkflowEventHandler<TrainOrderItem> trainOrderItemWorkflowEventHandler() {

        ImmutableMap.Builder<EOrderItemState,
                StatefulWorkflowEventHandler<EOrderItemState, TrainOrderItem>> builder = ImmutableMap.builder();
        builder
                .put(EOrderItemState.IS_NEW, new NewStateHandler(
                        billingPartnerService, trainWorkflowProperties, trainDiscountService))
                .put(EOrderItemState.IS_RESERVING, new ReservingStateHandler(imClientProvider,
                        trainWorkflowProperties, trainTariffInfoDataProvider, trainDiscountService, reserveCallLimiter))
                .put(EOrderItemState.IS_INSURANCE_PRICING_TRAINS,
                        new InsurancePricingStateHandler(imClientProvider,
                                trainMeters))
                .put(EOrderItemState.IS_CALCULATING_FEE_TRAINS, new CalculatingFeeStateHandler(
                        feeCalculationService, trainWorkflowProperties, starTrekService,
                        rebookingService, trainOrderItemRepository, trainFinancialDataProvider))
                .put(EOrderItemState.IS_RESERVED, new ReservedStateHandler())
                .put(EOrderItemState.IS_RESERVING_INSURANCE, new ReservingInsuranceStateHandler(
                        imClientProvider, starTrekService, trainWorkflowProperties, trainMeters))
                .put(EOrderItemState.IS_CONFIRMING, new ConfirmingStateHandler(deduplicationService,
                        imClientProvider, trainWorkflowProperties))
                .put(EOrderItemState.IS_CHECKING_CONFIRMATION_TRAINS, new CheckingConfirmationStateHandler(
                        starTrekService, trainDiscountService))
                .put(EOrderItemState.IS_CANCELLING_INSURANCE, new CancellingInsuranceStateHandler(
                        trainInsuranceRefundRepository, workflowRepository))
                .put(EOrderItemState.IS_CANCELLING, new CancellingStateHandler(imClientProvider,
                        trainWorkflowProperties,
                        trainDiscountService))
                .put(EOrderItemState.IS_CANCELLED, new CancelledStateHandler())
                .put(EOrderItemState.IS_REFUNDED, new RefundedStateHandler(imClientProvider))
                .put(EOrderItemState.IS_CONFIRMED, new ConfirmedStateHandler(trainTicketRefundRepository,
                        workflowRepository, orderRefundRepository, imClientProvider, trainWorkflowProperties,
                        trainDiscountService, trainInsuranceRefundRepository))
                .put(EOrderItemState.IS_REFUNDING, new RefundingTicketStateHandler(
                        trainTicketRefundRepository, trainDiscountService));

        return new ProxyStateMapWorkflowEventHandler<>(builder.build(), entityStateTransitionRepository);
    }


}
