package ru.yandex.travel.orders.workflows.orderitem.train.handlers;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.FiscalItem;
import ru.yandex.travel.orders.entities.FiscalItemType;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.management.StarTrekService;
import ru.yandex.travel.orders.repository.TrainOrderItemRepository;
import ru.yandex.travel.orders.services.FiscalTitleGenerator;
import ru.yandex.travel.orders.services.finances.providers.TrainFinancialDataProvider;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.services.train.FeeCalculationService;
import ru.yandex.travel.orders.services.train.RebookingService;
import ru.yandex.travel.orders.workflow.order.proto.TServiceReserved;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TCancellationCommit;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TFeeCalculationCommit;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TInsuranceReservationCommit;
import ru.yandex.travel.orders.workflows.orderitem.train.ImHelpers;
import ru.yandex.travel.orders.workflows.orderitem.train.TrainWorkflowProperties;
import ru.yandex.travel.train.model.ErrorCode;
import ru.yandex.travel.train.model.ErrorInfo;
import ru.yandex.travel.train.model.InsuranceStatus;
import ru.yandex.travel.train.model.TrainPassenger;
import ru.yandex.travel.train.model.TrainTicket;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
@RequiredArgsConstructor
public class CalculatingFeeStateHandler extends AnnotatedStatefulWorkflowEventHandler<EOrderItemState, TrainOrderItem> {
    private final FeeCalculationService feeCalculationService;
    private final TrainWorkflowProperties trainWorkflowProperties;
    private final StarTrekService starTrekService;
    private final RebookingService rebookingService;
    private final TrainOrderItemRepository trainOrderItemRepository;
    private final TrainFinancialDataProvider trainFinancialDataProvider;

    @HandleEvent
    public void handleFeeCalculation(TFeeCalculationCommit event, StateContext<EOrderItemState, TrainOrderItem> ctx) {
        TrainOrderItem orderItem = ctx.getWorkflowEntity();
        feeCalculationService.calculateAndAssignFees(orderItem, ctx.getAttempt());
        if (orderItem.getPayload().getIsRebookingFor() != null) {
            TrainOrderItem oldItem = trainOrderItemRepository.getOne(orderItem.getPayload().getIsRebookingFor());
            if (!rebookingService.checkRebookingValid(orderItem, oldItem)) {
                orderItem.getPayload().setErrorInfo(new ErrorInfo());
                orderItem.getPayload().getErrorInfo().setCode(ErrorCode.REBOOKING_FAILED);
                ctx.setState(EOrderItemState.IS_CANCELLING);
                ctx.scheduleEvent(TCancellationCommit.getDefaultInstance());
                return;
            }
            rebookingService.bindFiscalItems(orderItem, oldItem);
        } else {
            generateFiscalItems(orderItem, ctx);
        }
        if (orderItem.getPayload().getTargetInsuranceStatus() == InsuranceStatus.CHECKED_OUT) {
            ctx.setState(EOrderItemState.IS_RESERVING_INSURANCE);
            ctx.scheduleEvent(TInsuranceReservationCommit.newBuilder().build());
        } else {
            ctx.setState(EOrderItemState.IS_RESERVED);
            ctx.scheduleExternalEvent(ctx.getWorkflowEntity().getOrderWorkflowId(),
                    TServiceReserved.newBuilder().setServiceId(orderItem.getId().toString()).build());
        }
    }

    private void generateFiscalItems(TrainOrderItem orderItem, StateContext<EOrderItemState, TrainOrderItem> ctx) {
        int fiscalCounter = 0;
        boolean manyTrains = OrderCompatibilityUtils.getTrainOrderItems(orderItem.getOrder()).size() > 1;
        String trainNumber = orderItem.getPayload().getReservationRequestData().getTrainTicketNumber();
        for (TrainPassenger passenger : orderItem.getPayload().getPassengers()) {
            TrainTicket ticket = passenger.getTicket();
            if (ticket.calculateTotalCost().isZero()) {
                continue;
            }
            String inn;
            if (ticket.getCarrierInn() != null) {
                inn = ticket.getCarrierInn();
            } else {
                inn = trainWorkflowProperties.getBilling().getPartnerInn();
                starTrekService.createIssueForEmptyTrainCarrierInn(orderItem.getOrder(), ctx);
            }
            if (ticket.getTariffAmount().isPositive()) {
                ticket.setTariffFiscalItemInternalId(fiscalCounter++);
                var fiscalItem = new FiscalItem();
                fiscalItem.setInternalId(ticket.getTariffFiscalItemInternalId());
                fiscalItem.setMoneyAmount(ticket.getTariffAmount());
                fiscalItem.setVatType(ImHelpers.vatFromRate(ticket.getTariffVatRate()));
                fiscalItem.setTitle(FiscalTitleGenerator.makeTrainFiscalItemTitle(
                        trainWorkflowProperties.getBilling().getTicketFiscalTitle(), ticket.getPlaces(),
                        manyTrains, trainNumber));
                fiscalItem.setType(FiscalItemType.TRAIN_TICKET);
                fiscalItem.setInn(inn);
                orderItem.addFiscalItem(fiscalItem);
            }
            if (ticket.getServiceAmount().isPositive()) {
                ticket.setServiceFiscalItemInternalId(fiscalCounter++);
                var fiscalItem = new FiscalItem();
                fiscalItem.setInternalId(ticket.getServiceFiscalItemInternalId());
                fiscalItem.setMoneyAmount(ticket.getServiceAmount());
                fiscalItem.setVatType(ImHelpers.vatFromRate(ticket.getServiceVatRate()));
                fiscalItem.setTitle(FiscalTitleGenerator.makeTrainFiscalItemTitle(
                        trainWorkflowProperties.getBilling().getServiceFiscalTitle(), ticket.getPlaces(),
                        manyTrains, trainNumber));
                fiscalItem.setType(FiscalItemType.TRAIN_SERVICE);
                fiscalItem.setInn(inn);
                orderItem.addFiscalItem(fiscalItem);
            }
            if (ticket.getFeeAmount().isPositive()) {
                ticket.setFeeFiscalItemInternalId(fiscalCounter++);
                var fiscalItem = new FiscalItem();
                fiscalItem.setInternalId(ticket.getFeeFiscalItemInternalId());
                fiscalItem.setMoneyAmount(ticket.getFeeAmount());
                fiscalItem.setVatType(trainFinancialDataProvider.getYandexFeeVat(orderItem));
                fiscalItem.setTitle(FiscalTitleGenerator.makeTrainFiscalItemTitle(
                        trainWorkflowProperties.getBilling().getFeeFiscalTitle(), ticket.getPlaces(),
                        manyTrains, trainNumber));
                fiscalItem.setType(FiscalItemType.TRAIN_FEE);
                orderItem.addFiscalItem(fiscalItem);
            }
        }
    }
}
