package ru.yandex.travel.orders.workflows.orderitem.train.handlers;

import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.TrainInsuranceRefund;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.repository.TrainInsuranceRefundRepository;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.orderitem.train.insurancerefund.proto.TStartRefund;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TCancelInsurance;
import ru.yandex.travel.orders.workflow.train.proto.TServiceConfirmedWithoutInsurance;
import ru.yandex.travel.train.model.InsuranceStatus;
import ru.yandex.travel.train.model.refund.InsuranceItemInfo;
import ru.yandex.travel.train.partners.im.model.orderinfo.ImOperationStatus;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

@Slf4j
@RequiredArgsConstructor
public class CancellingInsuranceStateHandler
        extends AnnotatedStatefulWorkflowEventHandler<EOrderItemState, TrainOrderItem> {
    private final TrainInsuranceRefundRepository trainInsuranceRefundRepository;
    private final WorkflowRepository workflowRepository;

    @HandleEvent
    public void handleCancelInsurance(TCancelInsurance event,
                                      StateContext<EOrderItemState, TrainOrderItem> ctx) {
        TrainOrderItem orderItem = ctx.getWorkflowEntity();
        List<InsuranceItemInfo> itemsToRefund = orderItem.getPayload().getPassengers().stream()
                .filter(x -> x.getInsurance().getPartnerOperationStatus() != ImOperationStatus.FAILED)
                .map(x -> new InsuranceItemInfo(x.getInsurance().getPartnerOperationId(),
                        x.getInsurance().getPartnerOperationStatus()))
                .collect(Collectors.toList());
        UUID insuranceRefundId = null;
        if (itemsToRefund.size() > 0) {
            TrainInsuranceRefund refund = TrainInsuranceRefund.createInsuranceAutoRefund(orderItem, itemsToRefund);
            Workflow workflow = Workflow.createWorkflowForEntity(refund, ctx.getWorkflowEntity());
            workflow = workflowRepository.save(workflow);
            refund = trainInsuranceRefundRepository.save(refund);
            insuranceRefundId = refund.getId();
            ctx.scheduleExternalEvent(workflow.getId(), TStartRefund.newBuilder().build());
        }
        orderItem.getPayload().setInsuranceStatus(InsuranceStatus.AUTO_RETURN);
        ctx.setState(EOrderItemState.IS_CONFIRMED);
        var msg = TServiceConfirmedWithoutInsurance.newBuilder()
                .setServiceId(orderItem.getId().toString());
        if (insuranceRefundId != null) {
            msg.setInsuranceRefundId(insuranceRefundId.toString());
        }
        ctx.scheduleExternalEvent(ctx.getWorkflowEntity().getOrderWorkflowId(), msg.build());
    }
}
