package ru.yandex.travel.orders.workflows.orderitem.train.handlers;

import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.services.train.ImClientProvider;
import ru.yandex.travel.orders.services.train.TrainDiscountService;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TCancellationCommit;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TReservationExpired;
import ru.yandex.travel.orders.workflows.orderitem.train.TrainWorkflowProperties;
import ru.yandex.travel.train.partners.im.ImClientException;
import ru.yandex.travel.train.partners.im.ImClientRetryableException;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;
import ru.yandex.travel.workflow.exceptions.RetryableException;

@Slf4j
@RequiredArgsConstructor
@IgnoreEvents(types = {
        ru.yandex.travel.orders.workflow.orderitem.generic.proto.TCancellationStart.class,
        ru.yandex.travel.orders.workflow.orderitem.train.proto.TCancellationStart.class,
        ru.yandex.travel.orders.workflow.orderitem.generic.proto.TConfirmationStart.class,
        ru.yandex.travel.orders.workflow.orderitem.train.proto.TConfirmationStart.class,
        TReservationExpired.class
})
public class CancellingStateHandler
    extends AnnotatedStatefulWorkflowEventHandler<EOrderItemState, TrainOrderItem> {

    private final ImClientProvider imClientProvider;
    private final TrainWorkflowProperties trainWorkflowProperties;
    private final TrainDiscountService trainDiscountService;

    @HandleEvent
    public void handleCancellationCommit(TCancellationCommit event,
                                         StateContext<EOrderItemState, TrainOrderItem> ctx) {
        TrainOrderItem orderItem = ctx.getWorkflowEntity();
        if (!orderItem.getPayload().isSlaveItem()) {
            try {
                var imClient = imClientProvider.getImClientForOrderItem(orderItem);
                imClient.reservationCancel(orderItem.getPayload().getPartnerOrderId());
            } catch (ImClientRetryableException e) {
                if (ctx.getAttempt() < trainWorkflowProperties.getCancellationMaxTries()) {
                    throw new RetryableException(e);
                } else {
                    logError(orderItem.getId(), e);
                }
            } catch (ImClientException e) {
                logError(orderItem.getId(), e);
            }
        }
        trainDiscountService.deleteDiscountsForOrder(orderItem);
        ctx.setState(EOrderItemState.IS_CANCELLED);
        ctx.scheduleExternalEvent(ctx.getWorkflowEntity().getOrderWorkflowId(),
            TServiceCancelled.newBuilder().setServiceId(orderItem.getId().toString()).build());
    }

    private void logError(UUID orderItemId, Exception e) {
        log.error("Error occurred while trying to cancel train order item {}", orderItemId, e);
    }
}
