package ru.yandex.travel.orders.workflows.orderitem.train.handlers;

import java.time.Instant;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.management.StarTrekService;
import ru.yandex.travel.orders.services.train.OrderItemPayloadService;
import ru.yandex.travel.orders.services.train.TrainDiscountService;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.orders.workflow.order.proto.TServiceConfirmed;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.EErrorCode;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TCancelInsurance;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TConfirmationFailed;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TConfirmationSuccess;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TPartnerInfoUpdated;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TReservationExpired;
import ru.yandex.travel.orders.workflow.train.proto.TServiceConfirmedWithoutInsurance;
import ru.yandex.travel.orders.workflows.orderitem.train.ImHelpers;
import ru.yandex.travel.orders.workflows.orderitem.train.TrainProtoMaps;
import ru.yandex.travel.train.model.ErrorInfo;
import ru.yandex.travel.train.model.InsuranceStatus;
import ru.yandex.travel.train.model.TrainPassenger;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.train.partners.im.model.orderinfo.ImOperationStatus;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderInfoResponse;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderItemResponse;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;

@Slf4j
@RequiredArgsConstructor
@IgnoreEvents(types = TReservationExpired.class)
public class CheckingConfirmationStateHandler extends AnnotatedStatefulWorkflowEventHandler<EOrderItemState, TrainOrderItem> {
    private final StarTrekService starTrekService;
    private final TrainDiscountService trainDiscountService;

    @HandleEvent
    public void handleConfirmationSuccess(TConfirmationSuccess event, StateContext<EOrderItemState, TrainOrderItem> ctx) {
        var orderInfoResponse = ProtoUtils.fromTJson(event.getPartnerResult(), OrderInfoResponse.class);
        handleConfirmationDone(null, orderInfoResponse, ctx);
    }

    @HandleEvent
    public void handleConfirmationFailed(TConfirmationFailed event, StateContext<EOrderItemState, TrainOrderItem> ctx) {
        OrderInfoResponse orderInfoResponse = null;
        ErrorInfo errorInfo = null;
        if (event.hasPartnerResult()) {
            orderInfoResponse = ProtoUtils.fromTJson(event.getPartnerResult(), OrderInfoResponse.class);
        }
        if (!(Strings.isNullOrEmpty(event.getErrorMessage()) && event.getErrorCode() == EErrorCode.EM_UNKNOWN)) {
            errorInfo = new ErrorInfo();
            errorInfo.setMessage(event.getErrorMessage());
            errorInfo.setCode(TrainProtoMaps.getErrorCode(event.getErrorCode()));
        }
        handleConfirmationDone(errorInfo, orderInfoResponse, ctx);
    }

    @HandleEvent
    public void handlePartnerInfoUpdated(TPartnerInfoUpdated event,
                                         StateContext<EOrderItemState, TrainOrderItem> ctx) {
        OrderInfoResponse orderInfoResponse = ProtoUtils.fromTJson(event.getPayload(), OrderInfoResponse.class);
        TrainOrderItem orderItem = ctx.getWorkflowEntity();
        TrainReservation trainReservation = orderItem.getReservation();

        OrderItemPayloadService.mergeReservation(trainReservation, orderInfoResponse);
    }

    private void handleConfirmationDone(ErrorInfo errorInfo, OrderInfoResponse orderInfoResponse,
                                       StateContext<EOrderItemState, TrainOrderItem> ctx) {
        TrainOrderItem orderItem = ctx.getWorkflowEntity();
        ImOperationStatus status = ImOperationStatus.FAILED;

        if (orderInfoResponse != null) {
            List<OrderItemResponse> buyItems = orderInfoResponse.findItems(orderItem.getPayload().getPartnerBuyOperationIds());
            if (buyItems.size() > 0) {
                List<ImOperationStatus> statuses = buyItems.stream().map(OrderItemResponse::getSimpleOperationStatus)
                        .distinct().collect(Collectors.toList());
                if (statuses.size() != 1) {
                    // TODO(ganintsev): handle different statuses TRAINS-6495
                    throw new RuntimeException("Some imOrderItems have different statuses");
                }
                status = statuses.get(0);
                storeDataFromOrderInfo(orderItem, orderInfoResponse, status);
            }
            storeInsuranceDataFromOrderInfo(orderItem, orderInfoResponse);
        }

        if (errorInfo != null) {
            orderItem.getPayload().setErrorInfo(errorInfo);
        }

        if (status == ImOperationStatus.IN_PROCESS) {
            log.warn("Train order confirmed with IN_PROCESS status, processed as FAILED");
            //TODO (mbobrov): think of flag signaling suspicious cancel for future reconciliations
            // for the moment we consider that im order must change it's state from IN_PROCESS to OK or FAILED in
            // 15 minutes (confiugred in TrainOrderRefreshService), therefore if it doesn't happen we consider order
            // item to be cancelled
        }

        if (status == ImOperationStatus.OK) {
            if (orderItem.getPayload().getInsuranceStatus() == InsuranceStatus.CHECKED_OUT) {
                confirmOrderWithInsurance(orderItem, ctx);
            } else {
                confirmOrderWithoutInsurance(orderItem, ctx);
            }
        } else {
            log.info("Train reservation for TrainOrderItem {} is {}.", orderItem.getId(), status.getValue());
            trainDiscountService.deleteDiscountsForOrder(orderItem);
            ctx.setState(EOrderItemState.IS_CANCELLED);
            ctx.scheduleExternalEvent(ctx.getWorkflowEntity().getOrderWorkflowId(),
                    TServiceCancelled.newBuilder().setServiceId(orderItem.getId().toString()).build());
        }
    }

    private void confirmOrderWithoutInsurance(
            TrainOrderItem orderItem, StateContext<EOrderItemState, TrainOrderItem> ctx) {
        orderItem.setConfirmedAt(Instant.now());
        ctx.setState(EOrderItemState.IS_CONFIRMED);
        ctx.scheduleExternalEvent(ctx.getWorkflowEntity().getOrderWorkflowId(),
                TServiceConfirmed.newBuilder().setServiceId(orderItem.getId().toString()).build());
    }

    private void confirmOrderWithInsurance(
            TrainOrderItem orderItem, StateContext<EOrderItemState, TrainOrderItem> ctx) {
        boolean insuranceFailed = orderItem.getPayload().getPassengers().stream()
                .filter(p -> p.getInsurance() != null)
                .anyMatch(p -> p.getInsurance().getPartnerOperationStatus() == ImOperationStatus.FAILED);

        if (insuranceFailed) {
            cancelInsurance(orderItem, ctx);
        } else {
            boolean insurancePending = orderItem.getPayload().getPassengers().stream()
                    .filter(p -> p.getInsurance() != null)
                    .anyMatch(p -> p.getInsurance().getPartnerOperationStatus() == ImOperationStatus.IN_PROCESS);

            if (insurancePending) {
                // we continue with a warning to support
                starTrekService.createIssueForTrainInsuranceNotConfirmed(orderItem.getOrder(), ctx);
            }

            orderItem.setConfirmedAt(Instant.now());
            ctx.setState(EOrderItemState.IS_CONFIRMED);
            ctx.scheduleExternalEvent(ctx.getWorkflowEntity().getOrderWorkflowId(),
                    TServiceConfirmed.newBuilder().setServiceId(orderItem.getId().toString()).build());
        }
    }

    private void cancelInsurance(
            TrainOrderItem orderItem, StateContext<EOrderItemState, TrainOrderItem> ctx) {
        orderItem.setConfirmedAt(Instant.now());
        if (orderItem.getOrder().getPublicType() == EOrderType.OT_GENERIC) {
            ctx.setState(EOrderItemState.IS_CONFIRMED);
            var msg = TServiceConfirmedWithoutInsurance.newBuilder()
                    .setServiceId(orderItem.getId().toString());
            ctx.scheduleExternalEvent(ctx.getWorkflowEntity().getOrderWorkflowId(), msg.build());
        } else {
            ctx.setState(EOrderItemState.IS_CANCELLING_INSURANCE);
            ctx.scheduleEvent(TCancelInsurance.newBuilder().build());
        }
    }

    private void storeInsuranceDataFromOrderInfo(TrainOrderItem orderItem, OrderInfoResponse orderInfoResponse) {
        var insuranceOperations = orderInfoResponse.findBuyInsuranceItems().stream()
                .collect(Collectors.toMap(OrderItemResponse::getOrderItemId, x -> x));
        for (TrainPassenger passenger : orderItem.getPayload().getPassengers()) {
            var insurance = passenger.getInsurance();
            if (insurance != null && insurance.getPartnerOperationId() != null) {
                var operation = insuranceOperations.get(insurance.getPartnerOperationId());
                if (operation == null) {
                    insurance.setPartnerOperationStatus(ImOperationStatus.IN_PROCESS);
                } else {
                    insurance.setPartnerOperationStatus(operation.getSimpleOperationStatus());
                }
            }
        }
    }

    private void storeDataFromOrderInfo(TrainOrderItem orderItem, OrderInfoResponse orderInfoResponse, ImOperationStatus status) {
        if (status == ImOperationStatus.OK) {
            OrderItemResponse firstBuyItem = orderInfoResponse.findItem(
                    orderItem.getPayload().getPartnerBuyOperationIds().stream().min(Comparator.naturalOrder()).get());
            if (Strings.isNullOrEmpty(orderItem.getPayload().getReservationNumber()) ) {
                orderItem.getPayload().setReservationNumber(firstBuyItem.getReservationNumber());
            }
            orderItem.getPayload().setCanChangeElectronicRegistrationTill(ImHelpers.fromLocalDateTime(
                    firstBuyItem.getElectronicRegistrationExpirationDateTime(),
                    orderItem.getPayload().getStationFromRailwayTimezone()));
        }

        if ((status == ImOperationStatus.OK) || (status == ImOperationStatus.FAILED)) {
            OrderItemPayloadService.mergeReservation(orderItem.getPayload(), orderInfoResponse);
        }
    }
}
