package ru.yandex.travel.orders.workflows.orderitem.train.handlers;

import java.math.BigDecimal;
import java.time.Instant;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.io.BaseEncoding;
import com.google.protobuf.InvalidProtocolBufferException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.entities.FiscalItem;
import ru.yandex.travel.orders.entities.FiscalItemType;
import ru.yandex.travel.orders.entities.MoneyMarkup;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.OrderRefund;
import ru.yandex.travel.orders.entities.TrainInsuranceRefund;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.entities.TrainTicketRefund;
import ru.yandex.travel.orders.repository.OrderRefundRepository;
import ru.yandex.travel.orders.repository.TrainInsuranceRefundRepository;
import ru.yandex.travel.orders.repository.TrainTicketRefundRepository;
import ru.yandex.travel.orders.services.train.ImClientProvider;
import ru.yandex.travel.orders.services.train.OrderItemPayloadService;
import ru.yandex.travel.orders.services.train.TrainDiscountService;
import ru.yandex.travel.orders.workflow.order.proto.TServiceRefunded;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TChangeRegistrationStatus;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TInsuranceRefundStart;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TOfficeRefundOccured;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TPartnerInfoUpdated;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TRefundStart;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TReservationExpired;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TUpdateTickets;
import ru.yandex.travel.orders.workflow.orderitem.train.ticketrefund.proto.ETrainTicketRefundState;
import ru.yandex.travel.orders.workflow.orderitem.train.ticketrefund.proto.TStartRefund;
import ru.yandex.travel.orders.workflow.train.proto.TRegistrationStatusChanged;
import ru.yandex.travel.orders.workflow.train.proto.TServiceOfficeRefunded;
import ru.yandex.travel.orders.workflow.train.proto.TTrainRefundPassenger;
import ru.yandex.travel.orders.workflow.train.proto.TTrainRefundToken;
import ru.yandex.travel.orders.workflow.train.proto.TTrainTicketsUpdated;
import ru.yandex.travel.orders.workflows.orderitem.RefundingUtils;
import ru.yandex.travel.orders.workflows.orderitem.train.ImHelpers;
import ru.yandex.travel.orders.workflows.orderitem.train.TrainOrderItemHelpers;
import ru.yandex.travel.orders.workflows.orderitem.train.TrainWorkflowProperties;
import ru.yandex.travel.train.model.InsuranceStatus;
import ru.yandex.travel.train.model.PassengerCategory;
import ru.yandex.travel.train.model.TrainModelHelpers;
import ru.yandex.travel.train.model.TrainPassenger;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.train.model.TrainTicket;
import ru.yandex.travel.train.model.TrainTicketRefundStatus;
import ru.yandex.travel.train.model.refund.InsuranceItemInfo;
import ru.yandex.travel.train.model.refund.PassengerRefundInfo;
import ru.yandex.travel.train.partners.im.ImClientRetryableException;
import ru.yandex.travel.train.partners.im.model.ElectronicRegistrationRequest;
import ru.yandex.travel.train.partners.im.model.ElectronicRegistrationResponse;
import ru.yandex.travel.train.partners.im.model.ImBlankStatus;
import ru.yandex.travel.train.partners.im.model.orderinfo.ImOperationStatus;
import ru.yandex.travel.train.partners.im.model.orderinfo.ImOperationType;
import ru.yandex.travel.train.partners.im.model.orderinfo.ImOrderItemType;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderInfoResponse;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderItemBlank;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderItemResponse;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.exceptions.RetryableException;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

import static ru.yandex.travel.orders.workflows.orderitem.RefundingUtils.convertTargetFiscalItemsToCardOnlyMarkup;

@Slf4j
@RequiredArgsConstructor
@IgnoreEvents(types = TReservationExpired.class)
public class ConfirmedStateHandler
        extends AnnotatedStatefulWorkflowEventHandler<EOrderItemState, TrainOrderItem> {
    private final TrainTicketRefundRepository trainTicketRefundRepository;
    private final WorkflowRepository workflowRepository;
    private final OrderRefundRepository orderRefundRepository;
    private final ImClientProvider imClientProvider;
    private final TrainWorkflowProperties trainWorkflowProperties;
    private final TrainDiscountService trainDiscountService;
    private final TrainInsuranceRefundRepository trainInsuranceRefundRepository;

    @HandleEvent
    public static void handlePartnerInfoUpdated(TPartnerInfoUpdated event,
                                                StateContext<EOrderItemState, TrainOrderItem> ctx) {
        OrderInfoResponse orderInfoResponse = ProtoUtils.fromTJson(event.getPayload(), OrderInfoResponse.class);
        TrainOrderItem orderItem = ctx.getWorkflowEntity();
        TrainReservation trainReservation = orderItem.getReservation();

        OrderItemPayloadService.mergeReservation(trainReservation, orderInfoResponse);
    }

    @HandleEvent
    public void handleUpdateTickets(TUpdateTickets event, StateContext<EOrderItemState, TrainOrderItem> ctx) {
        handleUpdateTickets(imClientProvider, event, ctx);
    }

    public static void handleUpdateTickets(ImClientProvider imClientProvider, TUpdateTickets event,
                                           StateContext<EOrderItemState, TrainOrderItem> ctx) {
        var orderItem = ctx.getWorkflowEntity();
        var payload = orderItem.getPayload();

        var imClient = imClientProvider.getImClientForOrderItem(orderItem);

        OrderInfoResponse orderInfoResponse;
        try {
            for (var imOrderItemId : payload.getPartnerBuyOperationIds()) {
                imClient.updateBlanks(imOrderItemId);
            }
            orderInfoResponse = imClient.orderInfo(payload.getPartnerOrderId());
        } catch (ImClientRetryableException e) {
            throw new RetryableException(e);
        }
        OrderItemPayloadService.mergeReservation(payload, orderInfoResponse);

        ctx.scheduleExternalEvent(orderItem.getOrderWorkflowId(),
                TTrainTicketsUpdated.newBuilder()
                        .setServiceId(orderItem.getId().toString())
                        .build());
    }

    @HandleEvent
    public void handleRefundStart(TRefundStart event, StateContext<EOrderItemState, TrainOrderItem> ctx) {
        TTrainRefundToken token = getRefundToken(event.getToken());
        TrainOrderItem orderItem = ctx.getWorkflowEntity();
        TrainReservation trainReservation = orderItem.getReservation();

        Map<Integer, TrainPassenger> passengersByCustomerId = trainReservation.getPassengers().stream()
                .collect(Collectors.toMap(TrainPassenger::getCustomerId, x -> x));

        List<Integer> adultBlanks = trainReservation.getPassengers().stream()
                .filter(p -> p.getCategory() != PassengerCategory.BABY)
                .map(p -> p.getTicket().getBlankId()).collect(Collectors.toList());

        List<PassengerRefundInfo> itemsToRefund = token.getPassengerList().stream().map(x -> {
            TrainPassenger passenger = passengersByCustomerId.get(x.getCustomerId());
            TrainTicket ticket = passenger.getTicket();
            if (ticket.getRefundStatus() != null) {
                throw new RuntimeException(String.format("Unable to start refund ticket with RefundStatus=%s",
                        ticket.getRefundStatus()));
            }
            var refundInfo = new PassengerRefundInfo();
            refundInfo.setCustomerId(x.getCustomerId());
            refundInfo.setBlankId(x.getBlankId());
            refundInfo.setBuyOperationId(ticket.getPartnerBuyOperationId());
            refundInfo.setPassengerCategory(passenger.getCategory());
            refundInfo.setCheckDocumentNumber(passenger.getDocumentNumber());
            refundInfo.setCalculatedRefundTicketAmount(ProtoUtils.fromTPrice(x.getTicketRefundAmount()));
            refundInfo.setCalculatedRefundFeeAmount(ProtoUtils.fromTPrice(x.getFeeRefundAmount()));
            refundInfo.setCalculatedRefundInsuranceAmount(ProtoUtils.fromTPrice(x.getInsuranceRefundAmount()));
            refundInfo.setBlankStatus(ticket.getImBlankStatus());
            if (passenger.getCategory() == PassengerCategory.BABY && adultBlanks.contains(x.getBlankId())) {
                refundInfo.setDependent(true);
            }
            return refundInfo;
        }).collect(Collectors.toList());

        TrainOrderItemHelpers.setTrainTicketRefundStatusForBlanks(orderItem,
                token.getPassengerList().stream().map(TTrainRefundPassenger::getBlankId).collect(Collectors.toSet()),
                TrainTicketRefundStatus.REFUNDING);
        OrderRefund orderRefund = orderRefundRepository.getOne(UUID.fromString(event.getOrderRefundId()));
        TrainTicketRefund refund = TrainTicketRefund.createRefund(orderItem, itemsToRefund, orderRefund);
        Workflow workflow = Workflow.createWorkflowForEntity(refund, ctx.getWorkflowEntity());
        workflow = workflowRepository.save(workflow);
        trainTicketRefundRepository.save(refund);

        ctx.scheduleExternalEvent(workflow.getId(), TStartRefund.newBuilder().build());
        ctx.setState(EOrderItemState.IS_REFUNDING);
    }

    @HandleEvent
    public void handleChangeRegistrationStatus(TChangeRegistrationStatus event,
                                               StateContext<EOrderItemState, TrainOrderItem> ctx) {
        TrainOrderItem orderItem = ctx.getWorkflowEntity();
        boolean registrationEnabled = event.getEnabled();
        List<Integer> blankIdsForChange = event.getBlankIdsList();
        TrainReservation payload = orderItem.getPayload();
        Set<Integer> successBlankIds = new HashSet<>();

        var imClient = imClientProvider.getImClientForOrderItem(orderItem);
        try {
            Map<Integer, Set<Integer>> blankIdsByImOrderItemId = payload.getBlankIdsByImOrderItemId();
            for (var imOrderItemId : blankIdsByImOrderItemId.keySet()) {
                var request = new ElectronicRegistrationRequest();
                Set<Integer> itemBlankIdsForChange = blankIdsByImOrderItemId.get(imOrderItemId).stream()
                        .filter(blankIdsForChange::contains).collect(Collectors.toSet());
                if (itemBlankIdsForChange.size() == 0) {
                    continue;
                }
                request.setOrderItemId(imOrderItemId);
                request.setOrderItemBlankIds(List.copyOf(itemBlankIdsForChange));
                request.setSet(registrationEnabled);
                request.setSendNotification(false);
                ElectronicRegistrationResponse response = imClient.changeElectronicRegistration(request);

                if (response.getExpirationElectronicRegistrationDateTime() != null) {
                    successBlankIds.addAll(itemBlankIdsForChange);
                } else {
                    log.warn("Got null values in response to electronic registration change request");
                }
            }
        } catch (Exception e) {
            log.warn("Cannot get response for changing electronic registration", e);
        } finally {
            HashSet<Integer> blankIdsForChangeSet = new HashSet<>(blankIdsForChange);
            for (TrainPassenger passenger : payload.getPassengers()) {
                TrainTicket ticket = passenger.getTicket();
                if (blankIdsForChangeSet.contains(ticket.getBlankId())) {
                    if (!successBlankIds.contains(ticket.getBlankId())) {
                        ticket.setPendingElectronicRegistration(true);
                    } else {
                        ticket.setPendingElectronicRegistration(false);
                        ticket.setImBlankStatus(
                                registrationEnabled ? ImBlankStatus.REMOTE_CHECK_IN : ImBlankStatus.NO_REMOTE_CHECK_IN);
                    }
                }
            }

            ctx.scheduleExternalEvent(orderItem.getOrderWorkflowId(), TRegistrationStatusChanged.newBuilder()
                    .setServiceId(orderItem.getId().toString())
                    .build());
        }
    }

    private static TTrainRefundToken getRefundToken(String token) {
        try {
            return TTrainRefundToken.parseFrom(BaseEncoding.base64Url().decode(token));
        } catch (InvalidProtocolBufferException e) {
            throw new RuntimeException("Unable to deserialize refund token", e);
        }
    }

    @HandleEvent
    public void handleInsuranceRefund(TInsuranceRefundStart event, StateContext<EOrderItemState, TrainOrderItem> ctx) {
        TrainOrderItem orderItem = ctx.getWorkflowEntity();
        List<InsuranceItemInfo> itemsToRefund = orderItem.getPayload().getPassengers().stream()
                .filter(x -> x.getInsurance().getPartnerOperationStatus() != ImOperationStatus.FAILED)
                .map(x -> new InsuranceItemInfo(x.getInsurance().getPartnerOperationId(),
                        x.getInsurance().getPartnerOperationStatus()))
                .collect(Collectors.toList());
        UUID orderRefundId = UUID.fromString(event.getOrderRefundId());
        OrderRefund orderRefund = orderRefundRepository.getOne(orderRefundId);
        if (itemsToRefund.size() > 0) {
            TrainInsuranceRefund refund = TrainInsuranceRefund.createInsuranceRefund(orderItem, itemsToRefund,
                    orderRefund);
            Workflow workflow = Workflow.createWorkflowForEntity(refund, ctx.getWorkflowEntity());
            workflow = workflowRepository.save(workflow);
            trainInsuranceRefundRepository.save(refund);
            ctx.scheduleExternalEvent(workflow.getId(),
                    ru.yandex.travel.orders.workflow.orderitem.train.insurancerefund.proto.TStartRefund.newBuilder().build());
        }
        Map<Long, Money> targetFiscalItems = orderItem.getFiscalItems().stream()
                .filter(x -> x.getType() == FiscalItemType.TRAIN_INSURANCE)
                .collect(Collectors.toMap(FiscalItem::getFiscalItemId, ii -> Money.zero(ii.getCurrency())));
        Map<Long, MoneyMarkup> targetFiscalItemsMarkup = convertTargetFiscalItemsToCardOnlyMarkup(targetFiscalItems);
        orderItem.getPayload().setInsuranceStatus(InsuranceStatus.AUTO_RETURN);
        ctx.scheduleExternalEvent(orderItem.getOrderWorkflowId(), TServiceRefunded.newBuilder()
                .setServiceId(orderItem.getId().toString())
                .setOrderRefundId(orderRefundId.toString())
                .putAllTargetFiscalItems(RefundingUtils.convertTargetFiscalItemsToProto(targetFiscalItems))
                .putAllTargetFiscalItemsMarkup(RefundingUtils.convertTargetFiscalItemsMarkupToProto(targetFiscalItemsMarkup))
                .setServiceConfirmed(true)
                .build());
    }

    @RequiredArgsConstructor
    static private class OperationWithBlank {
        private final OrderItemResponse operation;
        private final OrderItemBlank blank;
    }

    @HandleEvent
    public void handleOfficeRefund(TOfficeRefundOccured event, StateContext<EOrderItemState, TrainOrderItem> ctx) {
        TrainOrderItem orderItem = ctx.getWorkflowEntity();
        TrainReservation trainReservation = orderItem.getReservation();
        String orderRefundId = event.getOrderRefundId();
        var serviceRefundedMessage = TServiceOfficeRefunded.newBuilder()
                .setOrderRefundId(orderRefundId)
                .setServiceId(orderItem.getId().toString());

        OrderRefund orderRefund = null;
        // TODO(tlg-13,ganintsev): TRAVELBACK-1828 event.getOrderRefundId() is null only for TrainOrder
        // ^ that's not true, need to find out the is going on here
        if (!Strings.isNullOrEmpty(orderRefundId)) {
            orderRefund = orderRefundRepository.getOne(UUID.fromString(orderRefundId));
        }
        var imClient = imClientProvider.getImClientForOrderItem(orderItem);
        OrderInfoResponse orderInfoResponse;
        try {
            for (var imOrderItemId : trainReservation.getPartnerBuyOperationIds()) {
                imClient.updateBlanks(imOrderItemId);
            }
            orderInfoResponse = imClient.orderInfo(trainReservation.getPartnerOrderId());
        } catch (ImClientRetryableException ex) {
            throw new RetryableException(ex);
        }

        Set<Integer> refundOperationIds = new HashSet<>(event.getRefundOperationIdsList());

        Map<Integer, List<OperationWithBlank>> imRefunds = new HashMap<>();
        Map<Integer, List<OrderItemResponse>> imInsuranceRefunds = new HashMap<>();
        for (OrderItemResponse item : orderInfoResponse.getOrderItems()) {
            if (item.getOperationType() != ImOperationType.REFUND) {
                continue;
            }
            if (item.getType() == ImOrderItemType.RAILWAY
                    && refundOperationIds.contains(item.getOrderItemId())
                    && trainReservation.getPartnerBuyOperationIds().contains(item.getPreviousOrderItemId())) {
                for (OrderItemBlank blank : item.getOrderItemBlanks()) {
                    imRefunds.computeIfAbsent(blank.getPreviousOrderItemBlankId(), k -> new ArrayList<>())
                            .add(new OperationWithBlank(item, blank));
                }
            } else if (item.getType() == ImOrderItemType.INSURANCE
                    && trainReservation.getInsuranceStatus() == InsuranceStatus.CHECKED_OUT) {
                imInsuranceRefunds.computeIfAbsent(item.getPreviousOrderItemId(), k -> new ArrayList<>())
                        .add(item);
            }
        }

        List<PassengerRefundInfo> blankRefunds = new ArrayList<>();
        for (TrainPassenger passenger : trainReservation.getPassengers()) {
            TrainTicket ticket = passenger.getTicket();
            if (!imRefunds.containsKey(ticket.getBlankId())) {
                continue;
            }
            for (var imOperationWithBlank : imRefunds.remove(ticket.getBlankId())) {
                PassengerRefundInfo blankRefund = getPassengerRefundInfo(
                        orderItem, passenger, ticket, imOperationWithBlank);
                if (trainReservation.getInsuranceStatus() == InsuranceStatus.CHECKED_OUT) {
                    var insuranceOperationId = passenger.getInsurance().getPartnerOperationId();
                    if (imInsuranceRefunds.containsKey(insuranceOperationId)) {
                        // all insurance refunds are processed within first operation because they are not linked to any
                        var imRefundOperations = imInsuranceRefunds.remove(insuranceOperationId);
                        blankRefund.setCalculatedRefundInsuranceAmount(Money.of(
                                imRefundOperations.stream()
                                        .map(OrderItemResponse::getAmount)
                                        .reduce(BigDecimal::add)
                                        .orElse(BigDecimal.ZERO),
                                ProtoCurrencyUnit.RUB
                        ));
                    }
                }
                blankRefunds.add(blankRefund);
            }
        }
        Preconditions.checkState(imRefunds.isEmpty(), "Some office refunds are not processed");
        Preconditions.checkState(!blankRefunds.isEmpty(), "Office refund blankRefunds is empty");
        var ticketRefund = TrainTicketRefund.createRefund(orderItem, blankRefunds, orderRefund);
        ticketRefund.setState(ETrainTicketRefundState.RS_REFUNDED);
        trainTicketRefundRepository.save(ticketRefund);
        Set<Integer> refundedBlanks = blankRefunds.stream()
                .map(PassengerRefundInfo::getBlankId)
                .collect(Collectors.toSet());
        TrainOrderItemHelpers.setTrainTicketRefundStatusForBlanks(orderItem,
                refundedBlanks, TrainTicketRefundStatus.REFUNDED);
        trainDiscountService.deleteDiscountsForOrderByBlanks(orderItem, refundedBlanks);
        Map<Long, Money> targetFiscalItems = TrainOrderItemHelpers.createRefundFiscalItems(ticketRefund, orderItem);
        Map<Long, MoneyMarkup> targetFiscalItemsMarkup =
                convertTargetFiscalItemsToCardOnlyMarkup(targetFiscalItems);
        serviceRefundedMessage.setTicketRefundId(ticketRefund.getId().toString())
                .putAllTargetFiscalItems(RefundingUtils.convertTargetFiscalItemsToProto(targetFiscalItems))
                .putAllTargetFiscalItemsMarkup(RefundingUtils.convertTargetFiscalItemsMarkupToProto(targetFiscalItemsMarkup));
        log.info("Office refund just started, ticketRefundId: {}", ticketRefund.getId());
        boolean allTicketsRefunded = TrainModelHelpers.checkAllTicketsRefunded(orderItem.getPayload());
        if (allTicketsRefunded) {
            ctx.setState(EOrderItemState.IS_REFUNDED);
        }
        serviceRefundedMessage.setServiceConfirmed(!allTicketsRefunded);
        ctx.scheduleExternalEvent(orderItem.getOrderWorkflowId(), serviceRefundedMessage.build());
    }

    private PassengerRefundInfo getPassengerRefundInfo(TrainOrderItem orderItem, TrainPassenger passenger,
                                                       TrainTicket ticket, OperationWithBlank imOperationWithBlank) {
        var blankRefund = new PassengerRefundInfo();
        OrderItemResponse imRefundOperation = imOperationWithBlank.operation;
        OrderItemBlank imRefundBlank = imOperationWithBlank.blank;

        blankRefund.setCustomerId(passenger.getCustomerId());
        blankRefund.setCheckDocumentNumber(passenger.getDocumentNumber());
        blankRefund.setBlankId(ticket.getBlankId());
        blankRefund.setBuyOperationId(ticket.getPartnerBuyOperationId());
        blankRefund.setPassengerCategory(passenger.getCategory());
        blankRefund.setRefundBlankId(imRefundBlank.getOrderItemBlankId());
        blankRefund.setRefundOperationId(imRefundOperation.getOrderItemId());
        blankRefund.setRefundOperationStatus(imRefundOperation.getSimpleOperationStatus());
        blankRefund.setBlankStatus(imRefundBlank.getBlankStatus());

        blankRefund.setActualRefundTicketAmount(Money.zero(ProtoCurrencyUnit.RUB));
        blankRefund.setCalculatedRefundFeeAmount(Money.zero(ProtoCurrencyUnit.RUB));
        blankRefund.setCalculatedRefundTicketAmount(Money.zero(ProtoCurrencyUnit.RUB));
        blankRefund.setCalculatedRefundInsuranceAmount(Money.zero(ProtoCurrencyUnit.RUB));
        if (passenger.getCategory() != PassengerCategory.BABY) {
            blankRefund.setActualRefundTicketAmount(Money.of(imRefundBlank.getAmount(), ProtoCurrencyUnit.RUB));
            if (shouldRefundFee(orderItem, imRefundOperation.getMskConfirmedAt())) {
                var feeRefundAmount = ticket.calculateRefundFeeAmount();
                if (feeRefundAmount.isPositive()) {
                    blankRefund.setCalculatedRefundFeeAmount(feeRefundAmount);
                }
            }
        }
        return blankRefund;
    }

    private boolean shouldRefundFee(OrderItem orderItem, LocalDateTime mskRefundedAt) {
        if (orderItem.getConfirmedAt() == null || mskRefundedAt == null) {
            return false;
        }
        Instant refundedAt = ImHelpers.fromLocalDateTime(mskRefundedAt, ImHelpers.MSK_TZ);
        return orderItem.getConfirmedAt().plus(trainWorkflowProperties.getRefund().getReturnFeeTime())
                .isAfter(refundedAt);
    }
}
