package ru.yandex.travel.orders.workflows.orderitem.train.handlers;

import java.time.Instant;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.services.DeduplicationService;
import ru.yandex.travel.orders.services.train.ImClientProvider;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TConfirmationCommit;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TReservationExpired;
import ru.yandex.travel.orders.workflows.orderitem.train.TrainWorkflowProperties;
import ru.yandex.travel.train.partners.im.ImClientException;
import ru.yandex.travel.train.partners.im.ImClientIOException;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;

@Slf4j
@RequiredArgsConstructor
@IgnoreEvents(types = TReservationExpired.class)
public class ConfirmingStateHandler
        extends AnnotatedStatefulWorkflowEventHandler<EOrderItemState, TrainOrderItem> {

    private final DeduplicationService deduplicationService;
    private final ImClientProvider imClientProvider;
    private final TrainWorkflowProperties trainWorkflowProperties;

    @HandleEvent
    public void handleConfirmationCommit(TConfirmationCommit event, StateContext<EOrderItemState, TrainOrderItem> ctx) {
        TrainOrderItem orderItem = ctx.getWorkflowEntity();
        if (tryRegisterAtMostOnceCall(UUID.fromString(event.getDeduplicationKey()))) {
            try {
                var imClient = imClientProvider.getImClientForOrderItem(orderItem);
                imClient.reservationConfirm(orderItem.getPayload().getPartnerOrderId());
            } catch (ImClientIOException e) {
                log.error("Communication error during booking confirmation", e);
            } catch (ImClientException e) {
                log.error("Error during booking confirmation", e);
            }
        } else {
            log.warn("Handler of TConfirmationCommit called twice. Skip IM confirmation.");
        }
        orderItem.setNextCheckConfirmedAt(Instant.now().plus(trainWorkflowProperties.getCheckConfirmationTryDelay().get(0)));
        orderItem.getPayload().setMaxPendingTill(orderItem.getNextCheckConfirmedAt());
        orderItem.setBackgroundJobActive(true);
        ctx.setState(EOrderItemState.IS_CHECKING_CONFIRMATION_TRAINS);
    }

    public boolean tryRegisterAtMostOnceCall(UUID deduplicationKey) {
        try {
            deduplicationService.registerAtMostOnceCall(deduplicationKey);
            return true;
        } catch (IllegalStateException e) {
            return false;
        }
    }
}
