package ru.yandex.travel.orders.workflows.orderitem.train.handlers;

import java.util.HashMap;
import java.util.Map;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.services.train.ImClientProvider;
import ru.yandex.travel.orders.services.train.TrainMeters;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TCancellationCommit;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TFeeCalculationCommit;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TInsurancePricingCommit;
import ru.yandex.travel.train.model.ErrorCode;
import ru.yandex.travel.train.model.ErrorInfo;
import ru.yandex.travel.train.model.Insurance;
import ru.yandex.travel.train.model.InsuranceStatus;
import ru.yandex.travel.train.model.TrainPassenger;
import ru.yandex.travel.train.partners.im.ImClient;
import ru.yandex.travel.train.partners.im.model.insurance.InsurancePricingRequest;
import ru.yandex.travel.train.partners.im.model.insurance.InsurancePricingResponse;
import ru.yandex.travel.train.partners.im.model.insurance.MainServiceReferenceInternal;
import ru.yandex.travel.train.partners.im.model.insurance.RailwayTravelPricingRequest;
import ru.yandex.travel.train.partners.im.model.insurance.RailwayTravelProductPricingInfo;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
@RequiredArgsConstructor
public class InsurancePricingStateHandler extends AnnotatedStatefulWorkflowEventHandler<EOrderItemState, TrainOrderItem> {
    private final ImClientProvider imClientProvider;
    private final TrainMeters trainMeters;

    @HandleEvent
    public void handleInsurancePricing(TInsurancePricingCommit event, StateContext<EOrderItemState, TrainOrderItem> ctx) {
        TrainOrderItem orderItem = ctx.getWorkflowEntity();
        ImClient imClient = imClientProvider.getImClientForOrderItem(orderItem);
        try {
            Map<Integer, RailwayTravelProductPricingInfo> pricingByCustomerId = new HashMap<>();
            for (var imOrderItemId : orderItem.getPayload().getPartnerBuyOperationIds()) {
                var request = new InsurancePricingRequest(new RailwayTravelPricingRequest(
                        new MainServiceReferenceInternal(imOrderItemId)));
                InsurancePricingResponse response = imClient.insurancePricing(request);
                Map<Integer, RailwayTravelProductPricingInfo> customersResult = response.getPricingResult().getProductPricingInfoList().stream()
                    .collect(Collectors.toMap(RailwayTravelProductPricingInfo::getOrderCustomerId, x -> x));
                pricingByCustomerId.putAll(customersResult);
            }
            for (TrainPassenger passenger : orderItem.getPayload().getPassengers()) {
                RailwayTravelProductPricingInfo result = pricingByCustomerId.get(passenger.getCustomerId());
                passenger.setInsurance(new Insurance());
                passenger.getInsurance().setAmount(Money.of(result.getAmount(), ProtoCurrencyUnit.RUB));
                passenger.getInsurance().setCompensation(Money.of(result.getCompensation(), ProtoCurrencyUnit.RUB));
                passenger.getInsurance().setCompany(result.getCompany());
                passenger.getInsurance().setProductPackage(result.getProductPackage());
                passenger.getInsurance().setProvider(result.getProvider());
            }
            orderItem.getPayload().setInsuranceStatus(InsuranceStatus.PRICED);
        } catch (Exception ex) {
            log.error("Error insurance pricing", ex);
            orderItem.getPayload().setInsuranceStatus(InsuranceStatus.PRICING_FAILED);
            if (orderItem.getPayload().getTargetInsuranceStatus() == InsuranceStatus.CHECKED_OUT) {
                log.error("Rebooking failed because insurance not priced", ex);
                trainMeters.getTrainOrdersRebookingInsuranceFailed().increment();
                orderItem.getPayload().setErrorInfo(new ErrorInfo());
                orderItem.getPayload().getErrorInfo().setCode(ErrorCode.REBOOKING_FAILED);
                ctx.setState(EOrderItemState.IS_CANCELLING);
                ctx.scheduleEvent(TCancellationCommit.getDefaultInstance());
                return;
            }
        }

        ctx.setState(EOrderItemState.IS_CALCULATING_FEE_TRAINS);
        ctx.scheduleEvent(TFeeCalculationCommit.newBuilder().build());
    }
}
