package ru.yandex.travel.orders.workflows.orderitem.train.handlers;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.services.partners.BillingPartnerService;
import ru.yandex.travel.orders.services.train.TrainDiscountService;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.train.proto.TReservationCommit;
import ru.yandex.travel.orders.workflows.orderitem.train.TrainWorkflowProperties;
import ru.yandex.travel.train.model.InsuranceStatus;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
@RequiredArgsConstructor
public class NewStateHandler extends AnnotatedStatefulWorkflowEventHandler<EOrderItemState, TrainOrderItem> {
    private final BillingPartnerService billingPartnerService;
    private final TrainWorkflowProperties trainWorkflowProperties;
    private final TrainDiscountService trainDiscountService;

    @HandleEvent
    public void handleReservationStart(ru.yandex.travel.orders.workflow.orderitem.generic.proto.TReservationStart message,
                                       StateContext<EOrderItemState, TrainOrderItem> context) {
        reservationStart(context);
    }

    @HandleEvent
    public void handleTrainReservationStart(ru.yandex.travel.orders.workflow.orderitem.train.proto.TReservationStart message,
                                            StateContext<EOrderItemState, TrainOrderItem> context) {
        reservationStart(context);
    }

    private void reservationStart(StateContext<EOrderItemState, TrainOrderItem> context) {
        TrainOrderItem orderItem = context.getWorkflowEntity();
        billingPartnerService.addAgreementOrThrow(orderItem);

        if (orderItem.getPayload().getIsRebookingFor() == null) {
            if (trainWorkflowProperties.isInsuranceEnabled()) {
                orderItem.getPayload().setInsuranceStatus(InsuranceStatus.NEW);
            } else {
                orderItem.getPayload().setInsuranceStatus(InsuranceStatus.DISABLED);
            }
            if (trainWorkflowProperties.isSeparatePassengersReserving() &&
                    orderItem.getPayload().getPassengers().stream()
                            .anyMatch(x -> x.getRequestedPlaces() != null && x.getRequestedPlaces().size() > 0)) {
                orderItem.getPayload().setSeparatePassengersReserving(true);
            }
        }
        if (trainWorkflowProperties.isCheckPassengerDiscountsEnabled() && !orderItem.getPayload().isSlaveItem()) {
            trainDiscountService.checkAndStoreUniqueDiscounts(orderItem);
        }

        context.setState(EOrderItemState.IS_RESERVING);
        if (!orderItem.getPayload().isSlaveItem()) {
            context.scheduleEvent(TReservationCommit.newBuilder().build());
            log.info("TrainOrderItem '{}': scheduled reservation", context.getWorkflowEntity().getId());
        }
    }
}
