package ru.yandex.travel.orders.workflows.orderitem.train.handlers;

import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.entities.MoneyMarkup;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.entities.TrainTicketRefund;
import ru.yandex.travel.orders.repository.TrainTicketRefundRepository;
import ru.yandex.travel.orders.services.train.OrderItemPayloadService;
import ru.yandex.travel.orders.services.train.TrainDiscountService;
import ru.yandex.travel.orders.workflow.order.proto.TServiceRefundFailed;
import ru.yandex.travel.orders.workflow.order.proto.TServiceRefunded;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TPartnerInfoUpdated;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TRefundingTicketFailed;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TRefundingTicketSuccess;
import ru.yandex.travel.orders.workflows.orderitem.RefundingUtils;
import ru.yandex.travel.orders.workflows.orderitem.train.TrainOrderItemHelpers;
import ru.yandex.travel.train.model.TrainModelHelpers;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.train.model.TrainTicketRefundStatus;
import ru.yandex.travel.train.model.refund.PassengerRefundInfo;
import ru.yandex.travel.train.partners.im.model.orderinfo.ImOperationStatus;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderInfoResponse;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

import static ru.yandex.travel.orders.workflows.orderitem.RefundingUtils.convertTargetFiscalItemsToCardOnlyMarkup;

@Slf4j
@RequiredArgsConstructor
public class RefundingTicketStateHandler extends AnnotatedStatefulWorkflowEventHandler<EOrderItemState, TrainOrderItem> {
    private final TrainTicketRefundRepository trainTicketRefundRepository;
    private final TrainDiscountService trainDiscountService;

    @HandleEvent
    public void handleRefundingTicketFailed(TRefundingTicketFailed event,
                                            StateContext<EOrderItemState, TrainOrderItem> ctx) {
        TrainOrderItem orderItem = ctx.getWorkflowEntity();
        TrainOrderItemHelpers.setTrainTicketRefundStatusForBlanks(orderItem, Set.copyOf(event.getBlankIdsList()), null);
        TrainTicketRefund refund = trainTicketRefundRepository.getOne(UUID.fromString(event.getRefundId()));
        ctx.setState(EOrderItemState.IS_CONFIRMED);
        ctx.scheduleExternalEvent(orderItem.getOrderWorkflowId(), TServiceRefundFailed.newBuilder()
                .setServiceId(orderItem.getId().toString())
                .setOrderRefundId(refund.getOrderRefund().getId().toString())
                .build());
    }

    @HandleEvent
    public void handleRefundingTicketSuccess(TRefundingTicketSuccess event,
                                             StateContext<EOrderItemState, TrainOrderItem> ctx) {
        TrainOrderItem orderItem = ctx.getWorkflowEntity();
        TrainTicketRefund refund = trainTicketRefundRepository.getOne(UUID.fromString(event.getRefundId()));
        Set<Integer> refundedBlanks = refund.getPayload().getRefundedItems().stream()
                .map(PassengerRefundInfo::getBlankId)
                .collect(Collectors.toSet());
        TrainOrderItemHelpers.setTrainTicketRefundStatusForBlanks(orderItem, refundedBlanks, TrainTicketRefundStatus.REFUNDED);
        Set<Integer> refundFailedBlanks = refund.getPayload().getItems().stream()
                .filter(x -> x.getRefundOperationStatus() != ImOperationStatus.OK)
                .map(PassengerRefundInfo::getBlankId)
                .collect(Collectors.toSet());
        TrainOrderItemHelpers.setTrainTicketRefundStatusForBlanks(orderItem, refundFailedBlanks, null);
        Preconditions.checkState(orderItem.getPayload().getPassengers().stream()
                .noneMatch(x -> x.getTicket().getRefundStatus() == TrainTicketRefundStatus.REFUNDING),
                "Refund complete, but any ticket still refunding");
        trainDiscountService.deleteDiscountsForOrderByBlanks(orderItem, refundedBlanks);

        boolean allTicketsRefunded = TrainModelHelpers.checkAllTicketsRefunded(orderItem.getPayload());
        if (allTicketsRefunded) {
            ctx.setState(EOrderItemState.IS_REFUNDED);
        } else {
            ctx.setState(EOrderItemState.IS_CONFIRMED);
        }

        Map<Long, Money> targetFiscalItems = TrainOrderItemHelpers.createRefundFiscalItems(refund, orderItem);
        Map<Long, MoneyMarkup> targetFiscalItemsMarkup = convertTargetFiscalItemsToCardOnlyMarkup(targetFiscalItems);
        ctx.scheduleExternalEvent(orderItem.getOrderWorkflowId(), TServiceRefunded.newBuilder()
                .setServiceId(orderItem.getId().toString())
                .setOrderRefundId(refund.getOrderRefund().getId().toString())
                .putAllTargetFiscalItems(RefundingUtils.convertTargetFiscalItemsToProto(targetFiscalItems))
                .putAllTargetFiscalItemsMarkup(RefundingUtils.convertTargetFiscalItemsMarkupToProto(targetFiscalItemsMarkup))
                .setServiceConfirmed(!allTicketsRefunded)
                .build());
    }

    @HandleEvent
    public void handlePartnerInfoUpdated(TPartnerInfoUpdated event,
                                         StateContext<EOrderItemState, TrainOrderItem> ctx) {
        OrderInfoResponse orderInfoResponse = ProtoUtils.fromTJson(event.getPayload(), OrderInfoResponse.class);
        TrainOrderItem orderItem = ctx.getWorkflowEntity();
        TrainReservation trainReservation = orderItem.getReservation();

        OrderItemPayloadService.mergeReservation(trainReservation, orderInfoResponse);
    }
}
