package ru.yandex.travel.orders.workflows.orderitem.train.handlers;

import java.math.BigDecimal;
import java.util.Set;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.FiscalItem;
import ru.yandex.travel.orders.entities.FiscalItemType;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.entities.VatType;
import ru.yandex.travel.orders.management.StarTrekService;
import ru.yandex.travel.orders.services.FiscalTitleGenerator;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.services.train.ImClientProvider;
import ru.yandex.travel.orders.services.train.TrainMeters;
import ru.yandex.travel.orders.workflow.order.proto.TServiceReserved;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TCancellationCommit;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TInsuranceReservationCommit;
import ru.yandex.travel.orders.workflows.orderitem.train.TrainWorkflowProperties;
import ru.yandex.travel.train.model.ErrorCode;
import ru.yandex.travel.train.model.ErrorInfo;
import ru.yandex.travel.train.model.InsuranceStatus;
import ru.yandex.travel.train.partners.im.ImClientException;
import ru.yandex.travel.train.partners.im.ImClientRetryableException;
import ru.yandex.travel.train.partners.im.model.insurance.InsuranceCheckoutRequest;
import ru.yandex.travel.train.partners.im.model.insurance.MainServiceReferenceInternal;
import ru.yandex.travel.train.partners.im.model.insurance.RailwayInsuranceTravelCheckoutRequest;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.exceptions.RetryableException;

@Slf4j
@RequiredArgsConstructor
public class ReservingInsuranceStateHandler
        extends AnnotatedStatefulWorkflowEventHandler<EOrderItemState, TrainOrderItem> {
    public static final Set<Integer> NON_FATAL_INSURANCE_ERROR_CODES = Set.of(
            701, // "До отправления осталось недостаточно времени для покупки страховки"
            -1  // insurance is null ! REMOVE THIS !
    );

    private final ImClientProvider imClientProvider;
    private final StarTrekService starTrekService;
    private final TrainWorkflowProperties trainWorkflowProperties;
    private final TrainMeters trainMeters;

    @HandleEvent
    public void handleInsuranceReservationCommit(TInsuranceReservationCommit event,
                                                 StateContext<EOrderItemState, TrainOrderItem> ctx) {
        TrainOrderItem orderItem = ctx.getWorkflowEntity();
        try {
            for (var passenger : orderItem.getPayload().getPassengers()) {
                var insurance = passenger.getInsurance();
                // TODO(ganintsev): remove this check
                if (insurance == null) {
                    throw new ImClientException(-1, "insurance is null");
                }
                // TODO(ganintsev): remove deprecated, use only passenger.getTicket().getPartnerBuyOperationId()
                Integer partnerBuyOperationId = passenger.getTicket().getPartnerBuyOperationId() != null ?
                        passenger.getTicket().getPartnerBuyOperationId() :
                        orderItem.getPayload().getPartnerBuyOperationId();
                var mainService = new MainServiceReferenceInternal(partnerBuyOperationId);
                mainService.setOrderCustomerId(passenger.getCustomerId());
                var product = new RailwayInsuranceTravelCheckoutRequest(insurance.getProductPackage());
                InsuranceCheckoutRequest request = new InsuranceCheckoutRequest(mainService, product,
                        insurance.getProvider(), insurance.getCompany());
                var result = imClientProvider.getImClientForOrderItem(orderItem).insuranceCheckout(request);
                insurance.setPartnerOperationId(result.getOrderItemId());

                BigDecimal expectedAmount = insurance.getAmount().getNumberStripped();
                if (expectedAmount.compareTo(result.getAmount()) != 0) {
                    log.error("Obtained insurance amount {} instead expected {}",
                            result.getAmount(), insurance.getAmount());
                    starTrekService.createIssueForTrainInsuranceInvalidAmount(
                            orderItem.getOrder(), ctx, expectedAmount, result.getAmount());
                }
            }
            orderItem.getPayload().setInsuranceStatus(InsuranceStatus.CHECKED_OUT);
            if (orderItem.getPayload().getTargetInsuranceStatus() != InsuranceStatus.CHECKED_OUT) {
                generateInsuranceFiscalItems(orderItem);
            }
        } catch (ImClientException e) {
            if (e instanceof ImClientRetryableException
                    && ctx.getAttempt() + 1 < trainWorkflowProperties.getInsuranceCheckoutMaxTries()) {
                throw new RetryableException("Failed to check the insurance out but the error looks temporary", e);
            }
            orderItem.getPayload().setInsuranceStatus(InsuranceStatus.CHECKOUT_FAILED);
            // If error occurred on second passenger, we pay for first added insurance.
            // We can refund first insurance in future.
            if (orderItem.getPayload().getTargetInsuranceStatus() == InsuranceStatus.CHECKED_OUT) {
                log.error("Rebooking failed because insurance not added", e);
                trainMeters.getTrainOrdersRebookingInsuranceFailed().increment();
                orderItem.getPayload().setErrorInfo(new ErrorInfo());
                orderItem.getPayload().getErrorInfo().setCode(ErrorCode.REBOOKING_FAILED);
                ctx.setState(EOrderItemState.IS_CANCELLING);
                ctx.scheduleEvent(TCancellationCommit.getDefaultInstance());
                return;
            } else if (NON_FATAL_INSURANCE_ERROR_CODES.contains(e.getCode())) {
                log.info("A non-fatal exception has happened, ignoring it: code {}, message '{}', params={}",
                        e.getCode(), e.getMessage(), e.getMessageParams());
            } else {
                log.error("Error during insurance reservation", e);
                starTrekService.createIssueForTrainInsuranceNotAdded(orderItem.getOrder(), ctx);
            }
        }
        ctx.setState(EOrderItemState.IS_RESERVED);
        ctx.scheduleExternalEvent(ctx.getWorkflowEntity().getOrderWorkflowId(),
                TServiceReserved.newBuilder().setServiceId(orderItem.getId().toString()).build());
    }

    private void generateInsuranceFiscalItems(TrainOrderItem orderItem) {
        boolean manyTrains = OrderCompatibilityUtils.getTrainOrderItems(orderItem.getOrder()).size() > 1;
        String trainNumber = orderItem.getPayload().getReservationRequestData().getTrainTicketNumber();
        int fiscalCounter = orderItem.getFiscalItems().stream().map(FiscalItem::getInternalId).max(Integer::compareTo)
                .orElseThrow();
        for (var passenger : orderItem.getPayload().getPassengers()) {
            var ticket = passenger.getTicket();
            var insurance = passenger.getInsurance();
            if (insurance.getAmount().isPositive()) {
                fiscalCounter++;
                insurance.setFiscalItemInternalId(fiscalCounter);
                var fiscalItem = new FiscalItem();
                fiscalItem.setInternalId(fiscalCounter);
                fiscalItem.setMoneyAmount(insurance.getAmount());
                fiscalItem.setVatType(VatType.VAT_0);
                fiscalItem.setTitle(FiscalTitleGenerator.makeTrainFiscalItemTitle(
                        trainWorkflowProperties.getBilling().getInsuranceFiscalTitle(), ticket.getPlaces(),
                        manyTrains, trainNumber));
                fiscalItem.setType(FiscalItemType.TRAIN_INSURANCE);
                fiscalItem.setInn(trainWorkflowProperties.getBilling().getPartnerInn());
                orderItem.addFiscalItem(fiscalItem);
            }
        }
    }
}
