package ru.yandex.travel.orders.workflows.orderitem.train.handlers;

import java.math.BigDecimal;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.netflix.concurrency.limits.Limiter;
import lombok.Data;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.services.mock.MockImClient;
import ru.yandex.travel.orders.services.train.ImClientProvider;
import ru.yandex.travel.orders.services.train.ImReservationManager;
import ru.yandex.travel.orders.services.train.TrainDiscountService;
import ru.yandex.travel.orders.services.train.tariffinfo.TrainTariffInfoDataProvider;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TFeeCalculationCommit;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TInsurancePricingCommit;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TPartnerReservationResponse;
import ru.yandex.travel.orders.workflow.train.proto.TReservationCommit;
import ru.yandex.travel.orders.workflows.orderitem.train.ImHelpers;
import ru.yandex.travel.orders.workflows.orderitem.train.TrainOrderItemHelpers;
import ru.yandex.travel.orders.workflows.orderitem.train.TrainWorkflowProperties;
import ru.yandex.travel.train.model.ErrorInfo;
import ru.yandex.travel.train.model.InsuranceStatus;
import ru.yandex.travel.train.model.TrainPassenger;
import ru.yandex.travel.train.model.TrainPlace;
import ru.yandex.travel.train.model.TrainTicket;
import ru.yandex.travel.train.partners.im.ImClient;
import ru.yandex.travel.train.partners.im.ImClientException;
import ru.yandex.travel.train.partners.im.ImClientInvalidPassengerEmailException;
import ru.yandex.travel.train.partners.im.ImClientInvalidPassengerPhoneException;
import ru.yandex.travel.train.partners.im.ImClientParseException;
import ru.yandex.travel.train.partners.im.ImClientRetryableException;
import ru.yandex.travel.train.partners.im.model.OrderCreateReservationCustomerResponse;
import ru.yandex.travel.train.partners.im.model.RailwayPassengerResponse;
import ru.yandex.travel.train.partners.im.model.RailwayReservationBlankResponse;
import ru.yandex.travel.train.partners.im.model.RailwayReservationResponse;
import ru.yandex.travel.train.partners.im.model.ReservationCreateRequest;
import ru.yandex.travel.train.partners.im.model.ReservationCreateResponse;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.exceptions.RetryableException;

@Slf4j
@RequiredArgsConstructor
public class ReservingStateHandler extends AnnotatedStatefulWorkflowEventHandler<EOrderItemState, TrainOrderItem> {

    private final ImClientProvider imClientProvider;
    private final TrainWorkflowProperties trainWorkflowProperties;
    private final TrainTariffInfoDataProvider trainTariffInfoDataProvider;
    private final TrainDiscountService trainDiscountService;
    private final Limiter<Void> limiter;

    @HandleEvent
    public void doTrainReserving(TReservationCommit event, StateContext<EOrderItemState, TrainOrderItem> ctx) {
        TrainOrderItem orderItem = ctx.getWorkflowEntity();
        Optional<Limiter.Listener> optionalListener = limiter.acquire(null);
        try {
            if (optionalListener.isEmpty()) {
                log.warn("Couldn't acquire token from limiter. Call to IM overloaded. Assuming order item CANCELLED");
                var errorInfo = ImHelpers.createErrorInfo(new ImClientParseException("Restricted"));
                ctx.scheduleEvent(TPartnerReservationResponse.newBuilder()
                        .setErrorInfo(ProtoUtils.toTJson(errorInfo)).build());
                return;
            }

            List<TrainOrderItem> slaves = TrainOrderItemHelpers.getSlaves(orderItem);
            ReservationCreateRequest request = ImReservationManager.createRequest(
                    trainTariffInfoDataProvider, orderItem, slaves);
            ImClient imClient = imClientProvider.getImClientForOrderItem(orderItem);
            try {
                ReservationCreateResponse response = imClient.reservationCreate(request,
                        MockImClient.getTrainOrderData(orderItem, slaves));
                ctx.scheduleEvent(TPartnerReservationResponse.newBuilder()
                        .setPayload(ProtoUtils.toTJson(response)).build());
            } catch (ImClientException e) {
                if (e instanceof ImClientRetryableException &&
                        ctx.getAttempt() < trainWorkflowProperties.getReservationMaxTries()) {
                    throw new RetryableException(e, trainWorkflowProperties.getReservationRetryDelay());
                }
                if (ctx.getAttempt() < trainWorkflowProperties.getReservationMaxTries()) {
                    if (e instanceof ImClientInvalidPassengerPhoneException) {
                        processImClientInvalidPassengerPhoneException(orderItem, slaves);
                        ctx.scheduleEvent(TReservationCommit.newBuilder().build());
                        return;
                    } else if (e instanceof ImClientInvalidPassengerEmailException) {
                        processImClientInvalidPassengerEmailException(orderItem, slaves);
                        ctx.scheduleEvent(TReservationCommit.newBuilder().build());
                        return;
                    }
                }

                var errorInfo = ImHelpers.createErrorInfo(e);
                ctx.scheduleEvent(TPartnerReservationResponse.newBuilder()
                        .setErrorInfo(ProtoUtils.toTJson(errorInfo)).build());
            }
        } finally {
            // if we got a listener on acquire then we must release it.
            // always assume call to be a success, as there's a retry logic and we want the following behaviour:
            // we use AIMD limit here and on timeouts it'll cut the number of concurrent calls so if IM is timing out
            // we'll reduce the number of concurrent calls, and then raising the number of concurrent calls when
            // timings stabilize
            optionalListener.ifPresent(Limiter.Listener::onSuccess);
        }
    }

    @HandleEvent
    public void handleReservationResponse(TPartnerReservationResponse event, StateContext<EOrderItemState,
            TrainOrderItem> ctx) {
        TrainOrderItem orderItem = ctx.getWorkflowEntity();
        if (orderItem.getPayload().isMasterItem()) {
            for (var slaveItem : TrainOrderItemHelpers.getSlaves(orderItem)) {
                if (slaveItem.getState() != EOrderItemState.IS_RESERVING) {
                    // waiting for slaves after rebooking
                    throw new RetryableException("Not all slaves in reserving state");
                }
                ctx.scheduleExternalEvent(slaveItem.getWorkflow().getId(), event);
            }
        }
        if (event.getResultsCase() == TPartnerReservationResponse.ResultsCase.PAYLOAD) {
            ReservationCreateResponse response = ProtoUtils.fromTJson(event.getPayload(),
                    ReservationCreateResponse.class);
            saveResponse(orderItem, response);
            if (orderItem.getPayload().getInsuranceStatus() == InsuranceStatus.DISABLED) {
                ctx.setState(EOrderItemState.IS_CALCULATING_FEE_TRAINS);
                ctx.scheduleEvent(TFeeCalculationCommit.newBuilder().build());
            } else {
                ctx.setState(EOrderItemState.IS_INSURANCE_PRICING_TRAINS);
                ctx.scheduleEvent(TInsurancePricingCommit.newBuilder().build());
            }
        } else if (event.getResultsCase() == TPartnerReservationResponse.ResultsCase.ERRORINFO) {
            ErrorInfo errorInfo = ProtoUtils.fromTJson(event.getErrorInfo(), ErrorInfo.class);
            orderItem.getPayload().setErrorInfo(errorInfo);
            trainDiscountService.deleteDiscountsForOrder(orderItem);
            ctx.setState(EOrderItemState.IS_CANCELLED);
            ctx.scheduleExternalEvent(ctx.getWorkflowEntity().getOrderWorkflowId(),
                    TServiceCancelled.newBuilder().setServiceId(orderItem.getId().toString()).build());
        } else {
            throw new RuntimeException("Unknown TPartnerReservationResponse result case");
        }
    }

    @VisibleForTesting
    public void saveResponse(TrainOrderItem orderItem, ReservationCreateResponse response) {
        var payload = orderItem.getPayload();
        Set<Integer> reserveItemsIndexes = payload.getPassengers().stream().map(TrainPassenger::getPartnerItemIndex)
                .filter(Objects::nonNull).collect(Collectors.toSet());
        Map<Integer, RailwayReservationResponse> reserveItemsByIndex;
        RailwayReservationResponse firstReserveItem;
        if (reserveItemsIndexes.size() == 0) {
            // TODO(ganintsev): replace this case to precondition check reserveItemsIndexes.size() > 0
            reserveItemsByIndex = response.getReservationResults().stream()
                    .filter(x -> x.getIndex() == payload.getPartnerItemIndex())
                    .collect(Collectors.toMap(RailwayReservationResponse::getIndex, x -> x));
            firstReserveItem = reserveItemsByIndex.get(payload.getPartnerItemIndex());
        } else {
            reserveItemsByIndex = response.getReservationResults().stream()
                    .filter(x -> reserveItemsIndexes.contains(x.getIndex()))
                    .collect(Collectors.toMap(RailwayReservationResponse::getIndex, x -> x));
            firstReserveItem =
                    reserveItemsByIndex.get(reserveItemsIndexes.stream().min(Comparator.naturalOrder()).get());
        }
        Preconditions.checkState(reserveItemsByIndex.size() > 0, "Not found reservation response items for service");
        Map<Integer, OrderCreateReservationCustomerResponse> responseCustomerByIndex = response.getCustomers().stream()
                .collect(Collectors.toMap(OrderCreateReservationCustomerResponse::getIndex, c -> c));
        Map<Integer, RailwayPassengerResponse> responsePassengerByCustomerId = reserveItemsByIndex.values().stream()
                .flatMap(i -> i.getPassengers().stream())
                .collect(Collectors.toMap(RailwayPassengerResponse::getOrderCustomerId, p -> p));
        Map<Integer, RailwayReservationBlankResponse> responseBlankById = reserveItemsByIndex.values().stream()
                .flatMap(x -> x.getBlanks().stream())
                .collect(Collectors.toMap(RailwayReservationBlankResponse::getOrderItemBlankId, b -> b));

        Instant minConfirmTill = response.getReservationResults().stream()
                .filter(x -> x.getConfirmTill() != null)
                .map(x -> ImHelpers.fromLocalDateTime(x.getConfirmTill(), ImHelpers.MSK_TZ))
                .min(Comparator.comparing(Instant::getEpochSecond)).orElseThrow();
        // in accordance with existing code
        // https://a.yandex-team.ru/arc/trunk/arcadia/travel/rasp/train_api/train_partners/im/reserve_tickets
        // .py?rev=6388265#L425
        orderItem.setExpiresAt(minConfirmTill);

        // https://st.yandex-team.ru/TRAVELFRONT-2021#5e39216f1a82f12220f8b95f
        // first electronic blank reservNumber is the reservationNumber we need to display, it's said it won't change
        orderItem.getPayload().setReservationNumber(firstReserveItem.getBlanks().get(0).getFareInfo().getReservNumber());

        payload.setTrainNumber(firstReserveItem.getTrainNumber());
        payload.setArrivalTime(ImHelpers.fromLocalDateTime(firstReserveItem.getLocalArrivalDateTime(),
                payload.getStationToTimezone()));
        payload.setDepartureTime(ImHelpers.fromLocalDateTime(firstReserveItem.getLocalDepartureDateTime(),
                payload.getStationFromTimezone()));
        payload.setStationFromCode(firstReserveItem.getOriginStationCode());
        payload.getUiData().setStationFromPartnerTitle(firstReserveItem.getOriginStation());
        payload.setStationToCode(firstReserveItem.getDestinationStationCode());
        payload.getUiData().setStationToPartnerTitle(firstReserveItem.getDestinationStation());
        payload.setCarNumber(firstReserveItem.getCarNumber());
        payload.setCarType(firstReserveItem.getCarType());
        payload.setCarrier(firstReserveItem.getCarrier());
        payload.setSuburban(firstReserveItem.isSuburban());
        // полагаем что флаг OnlyFullReturnPossible от ИМ можно игнорировать если в item-е один бланк
        payload.setOnlyFullReturnPossible(reserveItemsByIndex.values().stream()
                .anyMatch(i -> i.isOnlyFullReturnPossible() && i.getBlanks().size() > 1));
        payload.setPartnerDescription(firstReserveItem.getTimeDescription());
        var notice = getPartnerNotice(firstReserveItem.getTimeDescription());
        payload.setTimeNotice(notice.getTimeNotice());
        payload.setSpecialNotice(notice.getSpecialNotice());

        List<TrainPassenger> passengers = orderItem.getReservation().getPassengers();
        for (int i = 0; i < passengers.size(); i++) {
            TrainPassenger passenger = passengers.get(i);
            var itemIndex = passenger.getPartnerItemIndex() == null ? payload.getPartnerItemIndex() :
                    passenger.getPartnerItemIndex();
            Preconditions.checkState(reserveItemsByIndex.containsKey(itemIndex),
                    String.format("Not found reservation response item by index=%d", itemIndex));
            RailwayReservationResponse reserveItem = reserveItemsByIndex.get(itemIndex);
            OrderCreateReservationCustomerResponse responseCustomer = responseCustomerByIndex.get(i);
            RailwayPassengerResponse responsePassenger =
                    responsePassengerByCustomerId.get(responseCustomer.getOrderCustomerId());
            RailwayReservationBlankResponse responseBlank =
                    responseBlankById.get(responsePassenger.getOrderItemBlankId());
            passenger.setCustomerId(responseCustomer.getOrderCustomerId());
            passenger.setNonRefundableTariff(responseBlank.isNonRefundableTariff());
            var ticket = new TrainTicket();
            passenger.setTicket(ticket);
            ticket.setPartnerBuyOperationId(reserveItem.getOrderItemId());
            ticket.setBookedTariffCode(trainTariffInfoDataProvider.getOptionalTariffCode(responseBlank.getTariffType()));
            ticket.setRawTariffType(responseBlank.getTariffType());
            ticket.setRawTariffName(trainTariffInfoDataProvider.getOptionalTariffTitle(ticket.getBookedTariffCode()));

            if (ticket.getRawTariffName() == null || ticket.getRawTariffName().isEmpty()) {
                if (responseBlank.getTariffInfo() != null) {
                    ticket.setRawTariffName(responseBlank.getTariffInfo().getTariffName());
                } else {
                    throw new RuntimeException("Cannot obtain value for rawTariffName");
                }
            }

            ticket.setBlankId(responseBlank.getOrderItemBlankId());
            if (responseBlank.getFareInfo() != null) {
                ticket.setCarrierInn(responseBlank.getFareInfo().getCarrierTin());
            }
            saveAmounts(ticket, responsePassenger, responseBlank);
            ticket.setPlaces(new ArrayList<>());
            for (var responsePlace : responsePassenger.getPlacesWithType()) {
                var place = new TrainPlace();
                ticket.getPlaces().add(place);
                place.setNumber(responsePlace.getNumber());
                place.setType(responsePlace.getType());
            }
        }
        payload.setPartnerOrderId(response.getOrderId());
        orderItem.setProviderId(String.valueOf(response.getOrderId()));
    }

    @VisibleForTesting
    @Data
    public static class PartnerNotice {
        private String timeNotice;
        private String specialNotice;
    }

    @VisibleForTesting
    public static PartnerNotice getPartnerNotice(String description) {
        var result = new PartnerNotice();
        if (Strings.isNullOrEmpty(description)) {
            return result;
        }
        List<String> timeParts = new ArrayList<>();
        List<String> specialParts = new ArrayList<>();
        for (String s : description.split("\\.")) {
            s = s.replaceAll(" +", " ").strip();
            if (s.startsWith("ВРЕМЯ ")) {
                timeParts.add(s);
            } else {
                specialParts.add(s);
            }
        }
        if (timeParts.size() > 0) {
            result.setTimeNotice(String.join(". ", timeParts));
        }
        if (specialParts.size() > 0) {
            result.setSpecialNotice(String.join(". ", specialParts));
        }
        return result;
    }

    private static void saveAmounts(TrainTicket ticket, RailwayPassengerResponse responsePassenger,
                                    RailwayReservationBlankResponse responseBlank) {
        // checking cost for passengers ticket, can be zero
        var amount = responsePassenger.getAmount();
        if (amount.compareTo(BigDecimal.ZERO) > 0) {
            // getting service cost from total subtracting extras and fees
            ticket.setTariffAmount(Money.of(amount.add(responseBlank.getServicePrice().negate()),
                    ProtoCurrencyUnit.RUB));
            ticket.setTariffVatAmount(Money.of(responseBlank.getVatRateValues().get(0).getValue(),
                    ProtoCurrencyUnit.RUB));
            ticket.setTariffVatRate(responseBlank.getVatRateValues().get(0).getRate());
            ticket.setServiceAmount(Money.of(responseBlank.getServicePrice(), ProtoCurrencyUnit.RUB));
            ticket.setServiceVatAmount(Money.of(responseBlank.getVatRateValues().get(1).getValue(),
                    ProtoCurrencyUnit.RUB));
            ticket.setServiceVatRate(responseBlank.getVatRateValues().get(1).getRate());
        }
    }

    private static void processImClientInvalidPassengerPhoneException(TrainOrderItem orderItem, List<TrainOrderItem> slaves) {
        List<TrainPassenger> passengers = orderItem.getPayload().getPassengers();
        passengers.forEach(p -> p.setUsePhoneForReservation(false));
        for (var item : slaves) {
            List<TrainPassenger> slavePassengers = item.getPayload().getPassengers();
            slavePassengers.forEach(p -> p.setUsePhoneForReservation(false));
        }
    }

    private static void processImClientInvalidPassengerEmailException(TrainOrderItem orderItem, List<TrainOrderItem> slaves) {
        List<TrainPassenger> passengers = orderItem.getPayload().getPassengers();
        passengers.forEach(p -> p.setUseEmailForReservation(false));
        for (var item : slaves) {
            List<TrainPassenger> slavePassengers = item.getPayload().getPassengers();
            slavePassengers.forEach(p -> p.setUseEmailForReservation(false));
        }
    }
}
