package ru.yandex.travel.orders.workflows.orderitem.train.insurancerefund.handlers;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.entities.TrainInsuranceRefund;
import ru.yandex.travel.orders.management.StarTrekService;
import ru.yandex.travel.orders.workflow.orderitem.train.insurancerefund.proto.ETrainInsuranceRefundState;
import ru.yandex.travel.orders.workflow.orderitem.train.insurancerefund.proto.TRefundingFailed;
import ru.yandex.travel.orders.workflow.orderitem.train.insurancerefund.proto.TRefundingSuccess;
import ru.yandex.travel.orders.workflows.orderitem.train.TrainOrderItemHelpers;
import ru.yandex.travel.train.model.refund.InsuranceItemInfo;
import ru.yandex.travel.train.partners.im.model.orderinfo.ImOperationStatus;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderInfoResponse;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderItemResponse;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
@RequiredArgsConstructor
public class CheckingRefundStateHandler
        extends AnnotatedStatefulWorkflowEventHandler<ETrainInsuranceRefundState, TrainInsuranceRefund> {
    private final StarTrekService starTrekService;

    @HandleEvent
    public void handleRefundingSuccess(TRefundingSuccess message,
                                       StateContext<ETrainInsuranceRefundState, TrainInsuranceRefund> context) {
        OrderInfoResponse orderInfoResponse = ProtoUtils.fromTJson(message.getPartnerResult(), OrderInfoResponse.class);
        TrainInsuranceRefund refund = context.getWorkflowEntity();
        storeRefundStatus(refund, orderInfoResponse);
        context.setState(ETrainInsuranceRefundState.RS_REFUNDED);
    }

    @HandleEvent
    public void handleRefundingFailed(TRefundingFailed message,
                                      StateContext<ETrainInsuranceRefundState, TrainInsuranceRefund> context) {
        TrainInsuranceRefund refund = context.getWorkflowEntity();
        if (message.hasPartnerResult()) {
            OrderInfoResponse orderInfoResponse = ProtoUtils.fromTJson(message.getPartnerResult(), OrderInfoResponse.class);
            storeRefundStatus(refund, orderInfoResponse);
        }
        starTrekService.createIssueForTrainInsuranceRefundFailed(refund.getOrderItem().getOrder(), refund, context);
        context.setState(ETrainInsuranceRefundState.RS_FAILED);
    }

    private void storeRefundStatus(TrainInsuranceRefund refund, OrderInfoResponse orderInfoResponse) {
        for (InsuranceItemInfo item : refund.getPayload().getItems()) {
            OrderItemResponse itemInfo = TrainOrderItemHelpers.findInsuranceRefundItem(orderInfoResponse, item);
            if (itemInfo == null) {
                item.setRefundOperationStatus(ImOperationStatus.IN_PROCESS);
            } else {
                item.setRefundOperationStatus(itemInfo.getSimpleOperationStatus());
                item.setRefundOperationId(itemInfo.getOrderItemId());
            }
        }
    }
}
