package ru.yandex.travel.orders.workflows.orderitem.train.insurancerefund.handlers;

import java.time.Instant;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.TrainInsuranceRefund;
import ru.yandex.travel.orders.services.train.ImClientProvider;
import ru.yandex.travel.orders.workflow.orderitem.train.insurancerefund.proto.ETrainInsuranceRefundState;
import ru.yandex.travel.orders.workflow.orderitem.train.insurancerefund.proto.TRefundOne;
import ru.yandex.travel.train.model.refund.InsuranceItemInfo;
import ru.yandex.travel.train.partners.im.ImClientException;
import ru.yandex.travel.train.partners.im.ImClientIOException;
import ru.yandex.travel.train.partners.im.model.insurance.InsuranceReturnRequest;
import ru.yandex.travel.train.partners.im.model.orderinfo.ImOperationStatus;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.exceptions.RetryableException;

@Slf4j
@RequiredArgsConstructor
public class RefundingStateHandler
        extends AnnotatedStatefulWorkflowEventHandler<ETrainInsuranceRefundState, TrainInsuranceRefund> {
    private static final int REFERENCE_ID_MAX_LENGTH = 32;
    private final ImClientProvider imClientProvider;

    @HandleEvent
    public void handleRefundOne(TRefundOne message,
                                StateContext<ETrainInsuranceRefundState, TrainInsuranceRefund> context) {
        TrainInsuranceRefund refund = context.getWorkflowEntity();
        InsuranceItemInfo insurance = getItemToReturn(refund);

        insurance.setRefundReferenceId(createReferenceId(refund, insurance));
        try {
            var returnRequest = new InsuranceReturnRequest(insurance.getBuyOperationId());
            returnRequest.setAgentReferenceId(insurance.getRefundReferenceId());
            var response = imClientProvider.getImClientForOrderItem(refund.getOrderItem()).insuranceReturn(returnRequest);
            insurance.setRefundOperationId(response.getOrderItemId());
        } catch (ImClientIOException e) {
            throw new RetryableException(e);
        } catch (ImClientException e) {
            logError(refund.getId(), insurance.getBuyOperationId(), e);
        }
        insurance.setRefundOperationStatus(ImOperationStatus.IN_PROCESS);

        if (getItemToReturn(refund) != null) {
            context.scheduleEvent(TRefundOne.newBuilder().build());
        } else {
            refund.setBackgroundJobActive(true);
            refund.setNextCheckAt(Instant.now());
            context.setState(ETrainInsuranceRefundState.RS_CHECKING_REFUND);
        }
    }

    private void logError(UUID refundId, Integer buyOperationId, Exception e) {
        log.error("Error occured while trying to refund insurance={}, refundId={}", buyOperationId, refundId, e);
    }

    private static InsuranceItemInfo getItemToReturn(TrainInsuranceRefund refund) {
        return refund.getPayload().getItems().stream().filter(x -> x.getRefundOperationStatus() == null)
                .findFirst().orElse(null);
    }

    private static String createReferenceId(TrainInsuranceRefund refund, InsuranceItemInfo insurance) {
        String result = refund.getId().toString() + ":" + insurance.getBuyOperationId().toString();
        if (result.length() > REFERENCE_ID_MAX_LENGTH) {
            result = result.substring(result.length() - REFERENCE_ID_MAX_LENGTH);
        }
        return result;
    }
}
