package ru.yandex.travel.orders.workflows.orderitem.train.jobs;

import java.time.Instant;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.entities.TrainInsuranceRefund;
import ru.yandex.travel.orders.repository.TrainInsuranceRefundRepository;
import ru.yandex.travel.orders.services.train.ImClientProvider;
import ru.yandex.travel.orders.workflow.orderitem.train.insurancerefund.proto.ETrainInsuranceRefundState;
import ru.yandex.travel.orders.workflow.orderitem.train.insurancerefund.proto.TRefundingFailed;
import ru.yandex.travel.orders.workflow.orderitem.train.insurancerefund.proto.TRefundingSuccess;
import ru.yandex.travel.orders.workflows.orderitem.train.TrainOrderItemHelpers;
import ru.yandex.travel.orders.workflows.orderitem.train.TrainWorkflowProperties;
import ru.yandex.travel.train.model.refund.InsuranceItemInfo;
import ru.yandex.travel.train.partners.im.ImClientException;
import ru.yandex.travel.train.partners.im.model.orderinfo.ImOperationStatus;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderInfoResponse;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderItemResponse;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.WorkflowMessageSender;

@Service
@Slf4j
@RequiredArgsConstructor
public class InsuranceRefundRefreshService {

    private final TrainInsuranceRefundRepository insuranceRefundRepository;

    private final WorkflowMessageSender workflowMessageSender;

    private final ImClientProvider imClientProvider;

    private final TrainWorkflowProperties trainWorkflowProperties;

    @TransactionMandatory
    public List<UUID> fetchInsuranceRefundsCheckingState(Set<UUID> excludedIds, int maxResultSize) {
        return insuranceRefundRepository.getInsuranceRefundsForCheck(Instant.now(),
                ETrainInsuranceRefundState.RS_CHECKING_REFUND, EWorkflowState.WS_RUNNING, excludedIds,
                PageRequest.of(0, maxResultSize));
    }

    @TransactionMandatory
    public long getInsuranceRefundCheckingCount(Set<UUID> excludedIds) {
        return insuranceRefundRepository.countInsuranceRefundsForCheck(Instant.now(),
                ETrainInsuranceRefundState.RS_CHECKING_REFUND, EWorkflowState.WS_RUNNING, excludedIds);
    }

    @TransactionMandatory
    public void checkInsuranceRefund(UUID insuranceRefundId) {
        TrainInsuranceRefund refund = insuranceRefundRepository.getOne(insuranceRefundId);
        try (var ignored = NestedMdc.forEntity(refund.getId(), refund.getEntityType())) {
            if (!refund.isBackgroundJobActive()) {
                log.info("Background job not active, returning");
                return;
            }

            boolean anyInProgress = false;
            boolean anyFailed = false;
            Exception error = null;
            OrderInfoResponse orderInfoResponse = null;
            try {
                orderInfoResponse =
                        imClientProvider.getImClientForOrderItem(refund.getOrderItem()).orderInfo(refund.getPayload().getPartnerOrderId());
                for (InsuranceItemInfo item : refund.getPayload().getItems()) {
                    OrderItemResponse itemInfo = TrainOrderItemHelpers.findInsuranceRefundItem(orderInfoResponse, item);
                    if (itemInfo == null || itemInfo.getSimpleOperationStatus() == ImOperationStatus.IN_PROCESS) {
                        anyInProgress = true;
                    } else if (itemInfo.getSimpleOperationStatus() == ImOperationStatus.FAILED) {
                        anyFailed = true;
                    }
                }
            } catch (ImClientException ex) {
                log.error("Exception occurred while get orderInfo for insuranceRefund {}", insuranceRefundId, ex);
                error = ex;
            }

            int checkCounter = refund.getPayload().getCheckRefundCounter();
            if (orderInfoResponse == null || anyInProgress) {
                if (checkCounter < trainWorkflowProperties.getCheckInsuranceRefundMaxTries()) {
                    refund.setNextCheckAt(Instant.now().plus(trainWorkflowProperties.getCheckInsuranceRefundDelay()));
                    refund.getPayload().setCheckRefundCounter(checkCounter + 1);
                } else {
                    handleFailedState(refund, orderInfoResponse, error);
                }
            } else if (anyFailed) {
                handleFailedState(refund, orderInfoResponse, null);
            } else {
                handleSuccessState(refund, orderInfoResponse);
            }
        }
    }

    private void handleFailedState(TrainInsuranceRefund refund, OrderInfoResponse orderInfo, Exception error) {
        refund.setBackgroundJobActive(false);
        refund.setNextCheckAt(null);
        var message = TRefundingFailed.newBuilder();
        if (error != null) {
            message.setErrorMessage(error.getMessage());
        }
        if (orderInfo != null) {
            message.setPartnerResult(ProtoUtils.toTJson(orderInfo));
        }
        workflowMessageSender.scheduleEvent(refund.getWorkflow().getId(), message.build());
    }

    private void handleSuccessState(TrainInsuranceRefund refund, OrderInfoResponse orderInfo) {
        refund.setBackgroundJobActive(false);
        refund.setNextCheckAt(null);
        var message = TRefundingSuccess.newBuilder();
        message.setPartnerResult(ProtoUtils.toTJson(orderInfo));
        workflowMessageSender.scheduleEvent(refund.getWorkflow().getId(), message.build());
    }
}
