package ru.yandex.travel.orders.workflows.orderitem.train.jobs;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.entities.TrainRefundedOperation;
import ru.yandex.travel.orders.management.StarTrekService;
import ru.yandex.travel.orders.repository.TrainOrderItemRepository;
import ru.yandex.travel.orders.repository.TrainRefundedOperationRepository;
import ru.yandex.travel.orders.services.OperationTypes;
import ru.yandex.travel.orders.services.train.GenericOfficeRefundStartService;
import ru.yandex.travel.orders.services.train.ImClientProvider;
import ru.yandex.travel.orders.workflows.orderitem.train.TrainWorkflowProperties;
import ru.yandex.travel.train.partners.im.model.orderlist.ImShortOrderInfo;
import ru.yandex.travel.train.partners.im.model.orderlist.ImShortOrderItem;
import ru.yandex.travel.train.partners.im.model.orderlist.ImShortOrderItemType;
import ru.yandex.travel.train.partners.im.model.orderlist.OrderListRequest;
import ru.yandex.travel.workflow.WorkflowProcessService;
import ru.yandex.travel.workflow.single_operation.SingleOperationService;

@Service
@Slf4j
@RequiredArgsConstructor
public class OfficeRefundCheckService {
    static final String TASK_KEY = "OFFICE_REFUND_TASK_KEY";

    private final ImClientProvider imClientProvider;
    private final TrainOrderItemRepository trainOrderItemRepository;
    private final TrainRefundedOperationRepository trainRefundedOperationRepository;
    private final TrainWorkflowProperties trainWorkflowProperties;
    private final StarTrekService starTrekService;
    private final SingleOperationService singleOperationService;
    private final WorkflowProcessService workflowProcessService;
    private final Clock clock;

    void sendOfficeRefundMessages(String taskKey) {
        Map<Integer, List<ImShortOrderItem>> imOrderIdsWithOperations = getOfficeRefundedImOrderIdsWithOperations();
        Set<Integer> operationIds = imOrderIdsWithOperations.values().stream()
                .flatMap(Collection::stream)
                .map(ImShortOrderItem::getOrderItemId)
                .collect(Collectors.toSet());
        Set<Integer> filteredOperationIds = trainRefundedOperationRepository.excludeRefundedOperationIds(operationIds);
        for (Map.Entry<Integer, List<ImShortOrderItem>> imOrderIdWithOperations : imOrderIdsWithOperations.entrySet()) {
            List<ImShortOrderItem> refundOperations = imOrderIdWithOperations.getValue().stream()
                    .filter(o -> filteredOperationIds.contains(o.getOrderItemId()))
                    .collect(Collectors.toList());
            if (!refundOperations.isEmpty()) {
                safeSendOfficeRefundMessage(imOrderIdWithOperations.getKey(), refundOperations);
            }
        }
    }

    private void safeSendOfficeRefundMessage(Integer imOrderId, List<ImShortOrderItem> refundOperations) {
        List<TrainOrderItem> orderItems = getOrderItems(imOrderId);
        Preconditions.checkState(
                !orderItems.isEmpty(),
                "TrainOrderItems for IM orderID %s not found",
                imOrderId
        );

        var order = orderItems.get(0).getOrder();
        Preconditions.checkState(
                order.getPublicType() == EOrderType.OT_GENERIC,
                "Only generic orders are supported"
        );

        Map<TrainOrderItem, Set<Integer>> servicesToRefundOperationIds = new HashMap<>();
        for (var service : orderItems) {
            Set<Integer> serviceRefundOperationIds = filterServiceRefundOperations(service, refundOperations).stream()
                    .map(ImShortOrderItem::getOrderItemId)
                    .collect(Collectors.toSet());
            if (!serviceRefundOperationIds.isEmpty()) {
                servicesToRefundOperationIds.put(service, serviceRefundOperationIds);
            }
        }
        Preconditions.checkState(
                !servicesToRefundOperationIds.isEmpty(),
                "TrainOrderItems for IM orderID %s and refund operations %s not found",
                imOrderId,
                refundOperations.stream()
                        .map(ImShortOrderItem::getOrderItemId)
                        .collect(Collectors.toSet())
        );
        try {
            scheduleOfficeRefundOperation(order, servicesToRefundOperationIds, getOperationDelay());
        } catch (Exception e) {
            starTrekService.createIssueForTrainOfficeRefundHandleError(order, e, workflowProcessService);
            log.error("Failed to send message to order {} for office refund", order.getId(), e);
        }
    }

    public void scheduleOfficeRefundOperation(Order order,
                                              Map<TrainOrderItem, Set<Integer>> servicesToRefundOperationIds,
                                              Duration scheduleDelay) {
        var data = new GenericOfficeRefundStartService.StartOfficeRefundData(
                order.getId(),
                servicesToRefundOperationIds.entrySet().stream()
                        .map(serviceToRefundOperationIds -> new GenericOfficeRefundStartService.Service(
                                serviceToRefundOperationIds.getKey().getId(),
                                serviceToRefundOperationIds.getValue(),
                                null
                        ))
                        .collect(Collectors.toList()));
        singleOperationService.scheduleOperation(
                "GenericTrainOfficeRefund" + Instant.now(clock).toEpochMilli(),
                OperationTypes.GENERIC_TRAINS_OFFICE_REFUND.getValue(),
                data,
                Instant.now().plus(scheduleDelay)
        );
        servicesToRefundOperationIds.values().stream()
                .flatMap(Collection::stream)
                .forEach(operationId ->
                        trainRefundedOperationRepository.save(new TrainRefundedOperation(operationId)));
    }

    private List<ImShortOrderItem> filterServiceRefundOperations(TrainOrderItem service,
                                                                 List<ImShortOrderItem> refundOperations) {
        Set<Integer> operationIds = new HashSet<>(service.getReservation().getPartnerBuyOperationIds());
        return refundOperations.stream()
                .filter(o -> operationIds.contains(o.getPreviousOrderItemId()))
                .collect(Collectors.toList());
    }

    public List<TrainOrderItem> getOrderItems(Integer imOrderId) {
        return trainOrderItemRepository.findAllByProviderId(String.valueOf(imOrderId));
    }

    private Map<Integer, List<ImShortOrderItem>> getOfficeRefundedImOrderIdsWithOperations() {
        Map<Integer, List<ImShortOrderItem>> result = new HashMap<>();
        LocalDateTime today = LocalDateTime.of(LocalDate.now(), LocalTime.MIDNIGHT);
        int daysToCheck = trainWorkflowProperties.getOfficeRefund().getDaysToCheck();
        for (int daysBehind = 0; daysBehind < daysToCheck; daysBehind++) {
            LocalDateTime day = today.minusDays(daysBehind);
            safeGetOfficeRefundedImOrderIdsWithOperations(day).forEach((imOrderId, refundOperations) ->
                    result.computeIfAbsent(imOrderId, k -> new ArrayList<>()).addAll(refundOperations)
            );
        }
        return result;
    }

    private Map<Integer, List<ImShortOrderItem>> safeGetOfficeRefundedImOrderIdsWithOperations(LocalDateTime checkDate) {
        Map<Integer, List<ImShortOrderItem>> result = new HashMap<>();
        try {
            for (ImShortOrderInfo imOrderInfo :
                    imClientProvider.getImClientForOrderItem(null).orderList(new OrderListRequest(checkDate)).getOrders()) {
                result.computeIfAbsent(imOrderInfo.getOrderId(), k -> new ArrayList<>()).addAll(
                        imOrderInfo.getOrderItems().stream()
                                .filter(ImShortOrderItem::getIsExternallyLoaded)
                                .filter(i -> i.getType() == ImShortOrderItemType.RAILWAY)
                                .collect(Collectors.toList())
                );
            }
        } catch (Exception e) {
            log.error("Failed to get data for office refund on {}", checkDate, e);
        }
        return result;
    }

    private Duration getOperationDelay() {
        Duration minDelay = trainWorkflowProperties.getOfficeRefund().getRefundDelay();
        Duration maxExtraDelay = trainWorkflowProperties.getOfficeRefund().getRefundExtraDelay();
        long rndExtraDelayMillis = ThreadLocalRandom.current().nextLong(maxExtraDelay.toMillis());
        return minDelay.plus(Duration.ofMillis(rndExtraDelayMillis));
    }
}
