package ru.yandex.travel.orders.workflows.orderitem.train.jobs;

import java.time.Instant;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import com.google.protobuf.Message;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.entities.TrainTicketRefund;
import ru.yandex.travel.orders.repository.TrainTicketRefundRepository;
import ru.yandex.travel.orders.services.train.ImClientProvider;
import ru.yandex.travel.orders.workflow.orderitem.train.ticketrefund.proto.ETrainTicketRefundState;
import ru.yandex.travel.orders.workflow.orderitem.train.ticketrefund.proto.TRefundingFailed;
import ru.yandex.travel.orders.workflow.orderitem.train.ticketrefund.proto.TRefundingSuccess;
import ru.yandex.travel.orders.workflows.orderitem.train.TrainWorkflowProperties;
import ru.yandex.travel.train.model.refund.PassengerRefundInfo;
import ru.yandex.travel.train.partners.im.ImClientException;
import ru.yandex.travel.train.partners.im.model.orderinfo.ImOperationStatus;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderInfoResponse;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderItemResponse;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.WorkflowMessageSender;

@Service
@Slf4j
@RequiredArgsConstructor
public class TicketRefundRefreshService {

    private final TrainTicketRefundRepository ticketRefundRepository;

    private final WorkflowMessageSender workflowMessageSender;

    private final ImClientProvider imClientProvider;

    private final TrainWorkflowProperties trainWorkflowProperties;

    @TransactionMandatory
    public List<UUID> fetchTicketRefundsCheckingState(Set<UUID> excludedIds, int maxResultSize) {
        return ticketRefundRepository.getTicketRefundsForCheck(Instant.now(),
                ETrainTicketRefundState.RS_CHECKING_REFUND, EWorkflowState.WS_RUNNING, excludedIds,
                PageRequest.of(0, maxResultSize));
    }

    @TransactionMandatory
    public long getTicketRefundCheckingCount(Set<UUID> excludedIds) {
        return ticketRefundRepository.countTicketRefundsForCheck(Instant.now(),
                ETrainTicketRefundState.RS_CHECKING_REFUND, EWorkflowState.WS_RUNNING, excludedIds);
    }

    @TransactionMandatory
    public void checkTicketRefund(UUID ticketRefundId) {
        TrainTicketRefund refund = ticketRefundRepository.getOne(ticketRefundId);
        try (var ignored = NestedMdc.forEntity(refund.getId(), refund.getEntityType())) {
            if (!refund.isBackgroundJobActive()) {
                log.info("Background job not active, returning");
                return;
            }

            boolean anyInProgress = false;
            boolean anySuccess = false;
            OrderInfoResponse orderInfoResponse = null;
            try {
                orderInfoResponse =
                        imClientProvider.getImClientForOrderItem(refund.getOrderItem()).orderInfo(refund.getPayload().getPartnerOrderId());
                for (PassengerRefundInfo item : refund.getPayload().getItems()) {
                    if (item.getRefundReferenceId() == null) {
                        continue;
                    }
                    OrderItemResponse itemInfo = orderInfoResponse.getOrderItems().stream()
                            .filter(x -> item.getRefundReferenceId().equals(x.getAgentReferenceId()))
                            .findFirst().orElse(null);
                    if (itemInfo == null || itemInfo.getSimpleOperationStatus() == ImOperationStatus.IN_PROCESS) {
                        anyInProgress = true;
                    } else if (itemInfo.getSimpleOperationStatus() == ImOperationStatus.OK) {
                        anySuccess = true;
                    }
                }
            } catch (ImClientException ex) {
                log.error("Exception occurred while get orderInfo for ticketRefund {}", ticketRefundId, ex);
            }

            int checkCounter = refund.getPayload().getCheckRefundCounter();
            if (orderInfoResponse == null || anyInProgress) {
                if (checkCounter < trainWorkflowProperties.getCheckTicketRefundMaxTries()) {
                    refund.setNextCheckAt(Instant.now().plus(trainWorkflowProperties.getCheckTicketRefundDelay()));
                    refund.getPayload().setCheckRefundCounter(checkCounter + 1);
                } else {
                    handleRefundDone(refund, orderInfoResponse, anySuccess);
                }
            } else {
                handleRefundDone(refund, orderInfoResponse, anySuccess);
            }
        }
    }

    private void handleRefundDone(TrainTicketRefund refund, OrderInfoResponse orderInfo, boolean anySuccess) {
        refund.setBackgroundJobActive(false);
        refund.setNextCheckAt(null);
        Message message;
        if (anySuccess) {
            var builder = TRefundingSuccess.newBuilder();
            builder.setPartnerResult(ProtoUtils.toTJson(orderInfo));
            message = builder.build();
        } else {
            var builder = TRefundingFailed.newBuilder();
            if (orderInfo != null) {
                builder.setPartnerResult(ProtoUtils.toTJson(orderInfo));
            }
            message = builder.build();
        }
        workflowMessageSender.scheduleEvent(refund.getWorkflow().getId(), message);
    }
}
