package ru.yandex.travel.orders.workflows.orderitem.train.jobs;

import java.time.Instant;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.repository.TrainOrderItemRepository;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TReservationExpired;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.WorkflowMessageSender;

@Service
@Slf4j
@RequiredArgsConstructor
public class TrainOrderItemExpiredService {

    private final TrainOrderItemRepository trainOrderItemRepository;

    private final WorkflowMessageSender workflowMessageSender;

    private final List<EOrderItemState> orderItemStates = List.of(EOrderItemState.IS_RESERVED);

    @TransactionMandatory
    public List<UUID> fetchExpiredOrderItems(Set<UUID> active, int maxResultSize) {
        return trainOrderItemRepository.getExpiredOrderItems(Instant.now(), orderItemStates, EWorkflowState.WS_RUNNING,
            active, PageRequest.of(0, maxResultSize));
    }

    @TransactionMandatory
    public long countExpiredOrderItems(Set<UUID> active) {
        return trainOrderItemRepository.countExpiredOrderItems(Instant.now(), orderItemStates,
            EWorkflowState.WS_RUNNING, active);
    }

    @TransactionMandatory
    public void startCancellation(UUID orderItemId) {
        TrainOrderItem trainOrderItem = trainOrderItemRepository.getOne(orderItemId);
        try (var ignored = NestedMdc.forEntity(trainOrderItem.getId(), trainOrderItem.getEntityType())) {
            if (trainOrderItem.isExpired()) {
                log.info("Already expired, returning");
                return;
            }

            log.info("OrderItem {} expired at {}", orderItemId, trainOrderItem.getExpiresAt());

            trainOrderItem.setExpired(true);
            var message = TReservationExpired.newBuilder();
            workflowMessageSender.scheduleEvent(trainOrderItem.getWorkflow().getId(), message.build());
        }
    }
}
