package ru.yandex.travel.orders.workflows.orderitem.train.jobs;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import com.google.protobuf.Message;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.repository.TrainOrderItemRepository;
import ru.yandex.travel.orders.services.train.ImClientProvider;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TConfirmationFailed;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TConfirmationSuccess;
import ru.yandex.travel.orders.workflows.orderitem.train.ImHelpers;
import ru.yandex.travel.orders.workflows.orderitem.train.TrainOrderItemHelpers;
import ru.yandex.travel.orders.workflows.orderitem.train.TrainProtoMaps;
import ru.yandex.travel.orders.workflows.orderitem.train.TrainWorkflowProperties;
import ru.yandex.travel.train.model.ErrorInfo;
import ru.yandex.travel.train.partners.im.ImClientIOException;
import ru.yandex.travel.train.partners.im.model.orderinfo.ImOperationStatus;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderInfoResponse;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderItemResponse;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.WorkflowMessageSender;

@Service
@Slf4j
@RequiredArgsConstructor
public class TrainOrderItemRefreshService {

    public static final Collection<EOrderItemState> CHECKING_CONFIRMATION_STATES = List.of(
            EOrderItemState.IS_CHECKING_CONFIRMATION_TRAINS
    );

    private final TrainOrderItemRepository trainOrderItemRepository;

    private final WorkflowMessageSender workflowMessageSender;

    private final ImClientProvider imClientProvider;

    private final TrainWorkflowProperties trainWorkflowProperties;

    @TransactionMandatory
    public List<UUID> fetchOrderItemsCheckingConfirmation(Set<UUID> active, int maxResultSize) {
        return trainOrderItemRepository.getOrderItemsForCheckReserved(Instant.now(),
                CHECKING_CONFIRMATION_STATES, EWorkflowState.WS_RUNNING, active,
                PageRequest.of(0, maxResultSize));
    }

    @TransactionMandatory
    public long getOrderItemsCheckingConfirmationCount(Set<UUID> active) {
        return trainOrderItemRepository.countOrderItemsForCheckReserved(Instant.now(),
                CHECKING_CONFIRMATION_STATES, EWorkflowState.WS_RUNNING, active);
    }

    @TransactionMandatory
    public void checkItemConfirmed(UUID orderItemId) {
        TrainOrderItem trainOrderItem = trainOrderItemRepository.getOne(orderItemId);
        try (var ignored = NestedMdc.forEntity(trainOrderItem.getId(), trainOrderItem.getEntityType())) {
            if (!trainOrderItem.isBackgroundJobActive()) {
                log.info("Background job not active, returning");
                return;
            }
            if (trainOrderItem.getPayload().isSlaveItem()) {
                log.warn("Is slave item, returning");
                return;
            }

            Map<Integer, ImOperationStatus> buyOperationStatusByItemId = new HashMap<>();
            boolean anyInsuranceInProgress = false;
            boolean anyTicketInProgress = false;
            ErrorInfo errorInfo = null;
            OrderInfoResponse orderInfo = null;

            List<TrainOrderItem> allItems = new ArrayList<>();
            allItems.add(trainOrderItem);
            allItems.addAll(TrainOrderItemHelpers.getSlaves(trainOrderItem));
            try {
                var imClient = imClientProvider.getImClientForOrderItem(trainOrderItem);
                orderInfo = imClient.orderInfo(trainOrderItem.getPayload().getPartnerOrderId());
                List<OrderItemResponse> buyOperations = orderInfo.findBuyRailwayItems();
                buyOperationStatusByItemId = buyOperations.stream().collect(Collectors
                        .toMap(OrderItemResponse::getOrderItemId, OrderItemResponse::getSimpleOperationStatus));
                anyInsuranceInProgress = orderInfo.findBuyInsuranceItems().stream()
                        .anyMatch(x -> x.getSimpleOperationStatus() == ImOperationStatus.IN_PROCESS);
            } catch (ImClientIOException ex) {
                throw ex;
            } catch (Exception ex) {
                log.error("Exception occurred while get orderInfo for orderItem {}", orderItemId, ex);
                errorInfo = ImHelpers.createErrorInfo(ex);
            }

            for (TrainOrderItem item : allItems) {
                for (var imOrderItemId : item.getPayload().getPartnerBuyOperationIds()) {
                    var status = buyOperationStatusByItemId.get(imOrderItemId);
                    if (status == null || status == ImOperationStatus.IN_PROCESS) {
                        anyTicketInProgress = true;
                        break;
                    }
                }
            }

            int checkConfirmationCounter = trainOrderItem.getPayload().getCheckConfirmationCounter();
            checkConfirmationCounter++;
            if ((anyTicketInProgress || anyInsuranceInProgress) &&
                    checkConfirmationCounter < trainWorkflowProperties.getCheckConfirmationTryDelay().size()) {
                Duration delay = trainWorkflowProperties.getCheckConfirmationTryDelay().get(checkConfirmationCounter);
                trainOrderItem.setNextCheckConfirmedAt(Instant.now().plus(delay));
                trainOrderItem.getPayload().setMaxPendingTill(trainOrderItem.getNextCheckConfirmedAt());
                trainOrderItem.getPayload().setCheckConfirmationCounter(checkConfirmationCounter);
            } else {
                checkingConfirmationDone(allItems, buyOperationStatusByItemId, orderInfo, errorInfo);
            }
        }
    }

    private void checkingConfirmationDone(List<TrainOrderItem> allItems,
                                          Map<Integer, ImOperationStatus> buyOperationStatusByItemId,
                                          OrderInfoResponse orderInfo,
                                          ErrorInfo errorInfo) {
        for (var item : allItems) {
            Message message;
            Set<ImOperationStatus> statuses = item.getPayload().getPartnerBuyOperationIds().stream()
                    .map(buyOperationStatusByItemId::get).collect(Collectors.toSet());
            if (statuses.size() == 1 && statuses.contains(ImOperationStatus.OK)) {
                message = TConfirmationSuccess.newBuilder().setPartnerResult(ProtoUtils.toTJson(orderInfo)).build();
            } else {
                var builder = TConfirmationFailed.newBuilder();
                if (errorInfo != null) {
                    builder.setErrorCode(TrainProtoMaps.getProtoErrorCode(errorInfo.getCode()));
                    builder.setErrorMessage(Strings.nullToEmpty(errorInfo.getMessage()));
                }
                if (orderInfo != null) {
                    builder.setPartnerResult(ProtoUtils.toTJson(orderInfo));
                }
                message = builder.build();
            }
            item.setBackgroundJobActive(false);
            item.setNextCheckConfirmedAt(null);
            workflowMessageSender.scheduleEvent(item.getWorkflow().getId(), message);
        }
    }
}
