package ru.yandex.travel.orders.workflows.orderitem.train.jobs;

import java.util.Collection;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.support.DefaultTransactionDefinition;

import ru.yandex.travel.orders.workflows.orderitem.train.TrainWorkflowProperties;
import ru.yandex.travel.spring.tx.ForcedRollbackTxManagerWrapper;
import ru.yandex.travel.task_processor.AbstractTaskKeyProvider;
import ru.yandex.travel.task_processor.RegularTaskKeyProvider;
import ru.yandex.travel.task_processor.TaskKeyProvider;
import ru.yandex.travel.task_processor.TaskProcessor;
import ru.yandex.travel.task_processor.TaskProcessorProperties;

@Configuration
@RequiredArgsConstructor
public class TrainOrderTaskProcessorsConfiguration {
    private final TrainOrderItemExpiredService trainOrderItemExpiredService;

    private final TrainOrderItemRefreshService trainOrderItemRefreshService;

    private final InsuranceRefundRefreshService insuranceRefundRefreshService;

    private final TicketRefundRefreshService ticketRefundRefreshService;

    private final ForcedRollbackTxManagerWrapper forcedRollbackTxManagerWrapper;

    private final TrainWorkflowProperties trainWorkflowProperties;

    @Bean
    public TaskProcessor<UUID> trainOrderItemCheckingConfirmationProcessor() {

        TaskKeyProvider<UUID> taskKeyProvider = new AbstractTaskKeyProvider<>() {
            @Override
            public Collection<UUID> getPendingTaskKeys(int maxResultSize) {
                return trainOrderItemRefreshService.fetchOrderItemsCheckingConfirmation(getLockedTaskKeys(),
                        maxResultSize);
            }

            @Override
            public long getPendingTasksCount() {
                return trainOrderItemRefreshService.getOrderItemsCheckingConfirmationCount(getLockedTaskKeys());
            }
        };
        DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
        transactionDefinition.setName("TrainCheckingConfirmation");
        transactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
        TransactionDefinition txDefinition = new DefaultTransactionDefinition(transactionDefinition);

        return new TaskProcessor<>(taskKeyProvider, trainOrderItemRefreshService::checkItemConfirmed,
                forcedRollbackTxManagerWrapper, txDefinition,
                TaskProcessorProperties.builder().name("TrainCheckingConfirmation")
                        .daemonPoolThreads(true).gracefulPoolShutdown(true)
                        .poolSize(trainWorkflowProperties.getCheckConfirmationTaskNumberOfItems())
                        .initialStartDelay(trainWorkflowProperties.getCheckConfirmationTaskInitialStartDelay())
                        .scheduleRate(trainWorkflowProperties.getCheckConfirmationTaskPeriod()).build());
    }

    @Bean
    public TaskProcessor<UUID> trainInsuranceRefundCheckingProcessor() {
        TaskKeyProvider<UUID> taskKeyProvider = new AbstractTaskKeyProvider<>() {
            @Override
            public Collection<UUID> getPendingTaskKeys(int maxResultSize) {
                return insuranceRefundRefreshService.fetchInsuranceRefundsCheckingState(getLockedTaskKeys(),
                        maxResultSize);
            }

            @Override
            public long getPendingTasksCount() {
                return insuranceRefundRefreshService.getInsuranceRefundCheckingCount(getLockedTaskKeys());
            }
        };
        DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
        transactionDefinition.setName("TrainInsuranceRefundChecking");
        transactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
        TransactionDefinition txDefinition = new DefaultTransactionDefinition(transactionDefinition);

        return new TaskProcessor<>(taskKeyProvider, insuranceRefundRefreshService::checkInsuranceRefund,
                forcedRollbackTxManagerWrapper, txDefinition,
                TaskProcessorProperties.builder().name("TrainInsuranceRefundChecking")
                        .daemonPoolThreads(true).gracefulPoolShutdown(true)
                        .poolSize(trainWorkflowProperties.getCheckInsuranceRefundTaskNumberOfItems())
                        .initialStartDelay(trainWorkflowProperties.getCheckInsuranceRefundTaskInitialStartDelay())
                        .scheduleRate(trainWorkflowProperties.getCheckInsuranceRefundTaskPeriod()).build());
    }

    @Bean
    public TaskProcessor<UUID> trainTicketRefundCheckingProcessor() {
        TaskKeyProvider<UUID> taskKeyProvider = new AbstractTaskKeyProvider<>() {
            @Override
            public Collection<UUID> getPendingTaskKeys(int maxResultSize) {
                return ticketRefundRefreshService.fetchTicketRefundsCheckingState(getLockedTaskKeys(),
                        maxResultSize);
            }

            @Override
            public long getPendingTasksCount() {
                return ticketRefundRefreshService.getTicketRefundCheckingCount(getLockedTaskKeys());
            }
        };
        DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
        transactionDefinition.setName("TrainTicketRefundChecking");
        transactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
        TransactionDefinition txDefinition = new DefaultTransactionDefinition(transactionDefinition);

        return new TaskProcessor<>(taskKeyProvider, ticketRefundRefreshService::checkTicketRefund,
                forcedRollbackTxManagerWrapper, txDefinition,
                TaskProcessorProperties.builder().name("TrainTicketRefundChecking")
                        .daemonPoolThreads(true).gracefulPoolShutdown(true)
                        .poolSize(trainWorkflowProperties.getCheckTicketRefundTaskNumberOfItems())
                        .initialStartDelay(trainWorkflowProperties.getCheckTicketRefundTaskInitialStartDelay())
                        .scheduleRate(trainWorkflowProperties.getCheckTicketRefundTaskPeriod()).build());
    }

    @Bean
    public TaskProcessor<UUID> trainOrderItemCheckingExpirationProcessor() {

        TaskKeyProvider<UUID> taskKeyProvider = new AbstractTaskKeyProvider<>() {
            @Override
            public Collection<UUID> getPendingTaskKeys(int maxResultSize) {
                return trainOrderItemExpiredService.fetchExpiredOrderItems(getLockedTaskKeys(), maxResultSize);
            }

            @Override
            public long getPendingTasksCount() {
                return trainOrderItemExpiredService.countExpiredOrderItems(getLockedTaskKeys());
            }
        };
        DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
        transactionDefinition.setName("TrainExpirationChecking");
        transactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
        TransactionDefinition txDefinition = new DefaultTransactionDefinition(transactionDefinition);

        return new TaskProcessor<>(taskKeyProvider, trainOrderItemExpiredService::startCancellation,
                forcedRollbackTxManagerWrapper, txDefinition,
                TaskProcessorProperties.builder().name("TrainExpirationChecking")
                        .daemonPoolThreads(true).gracefulPoolShutdown(true)
                        .poolSize(trainWorkflowProperties.getCheckExpirationTaskNumberOfItems())
                        .initialStartDelay(trainWorkflowProperties.getCheckExpirationTaskInitialStartDelay())
                        .scheduleRate(trainWorkflowProperties.getCheckExpirationTaskPeriod()).build());
    }

    @ConditionalOnProperty("train-workflow.office-refund.enabled")
    @Configuration
    @RequiredArgsConstructor
    public static class TrainOrderItemCheckOfficeRefundConfiguration {
        private final ForcedRollbackTxManagerWrapper forcedRollbackTxManagerWrapper;
        private final OfficeRefundCheckService officeRefundCheckService;
        private final TrainWorkflowProperties trainWorkflowProperties;

        @Bean
        public TaskProcessor<String> trainOrderItemCheckOfficeRefundTaskProcessor() {
            TaskKeyProvider<String> taskKeyProvider = new RegularTaskKeyProvider<>(OfficeRefundCheckService.TASK_KEY);
            DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
            transactionDefinition.setName("TrainOfficeRefundChecking");
            transactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
            transactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
            TransactionDefinition txDefinition = new DefaultTransactionDefinition(transactionDefinition);

            return new TaskProcessor<>(taskKeyProvider, officeRefundCheckService::sendOfficeRefundMessages,
                    forcedRollbackTxManagerWrapper, txDefinition,
                    TaskProcessorProperties.builder().name("TrainOfficeRefundChecking")
                            .daemonPoolThreads(true).gracefulPoolShutdown(true)
                            .poolSize(1)  // need only one worker
                            .initialStartDelay(trainWorkflowProperties.getOfficeRefund().getCheckTaskInitialStartDelay())
                            .scheduleRate(trainWorkflowProperties.getOfficeRefund().getCheckTaskPeriod()).build());
        }
    }
}
