package ru.yandex.travel.orders.workflows.orderitem.train.ticketrefund.handlers;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.entities.TrainTicketRefund;
import ru.yandex.travel.orders.management.StarTrekService;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TRefundingTicketFailed;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TRefundingTicketSuccess;
import ru.yandex.travel.orders.workflow.orderitem.train.ticketrefund.proto.ETrainTicketRefundState;
import ru.yandex.travel.orders.workflow.orderitem.train.ticketrefund.proto.TRefundingFailed;
import ru.yandex.travel.orders.workflow.orderitem.train.ticketrefund.proto.TRefundingSuccess;
import ru.yandex.travel.train.model.refund.PassengerRefundInfo;
import ru.yandex.travel.train.partners.im.model.orderinfo.ImOperationStatus;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderInfoResponse;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderItemBlank;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderItemResponse;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
@RequiredArgsConstructor
public class CheckingRefundStateHandler
        extends AnnotatedStatefulWorkflowEventHandler<ETrainTicketRefundState, TrainTicketRefund> {
    private final StarTrekService starTrekService;

    @HandleEvent
    public void handleRefundingSuccess(TRefundingSuccess message,
                                       StateContext<ETrainTicketRefundState, TrainTicketRefund> context) {
        TrainTicketRefund refund = context.getWorkflowEntity();
        OrderInfoResponse orderInfoResponse = ProtoUtils.fromTJson(message.getPartnerResult(), OrderInfoResponse.class);
        storeRefundStatus(refund, orderInfoResponse);
        context.setState(ETrainTicketRefundState.RS_REFUNDED);
        List<Integer> blankIds = refund.getPayload().getRefundedItems().stream()
                .map(PassengerRefundInfo::getBlankId)
                .collect(Collectors.toList());
        boolean partFailed = refund.getPayload().getItems().stream()
                .anyMatch(x -> x.getRefundOperationStatus() != ImOperationStatus.OK);
        if (partFailed) {
            starTrekService.createIssueForTrainTicketRefundFailed(refund.getOrderItem().getOrder(), refund, context);
        }
        context.scheduleExternalEvent(refund.getOrderItemWorkflowId(), TRefundingTicketSuccess.newBuilder()
                .setRefundId(refund.getId().toString())
                .addAllBlankIds(blankIds).build());
    }

    @HandleEvent
    public void handleRefundingFailed(TRefundingFailed message,
                                      StateContext<ETrainTicketRefundState, TrainTicketRefund> context) {
        TrainTicketRefund refund = context.getWorkflowEntity();
        if (message.hasPartnerResult()) {
            OrderInfoResponse orderInfoResponse = ProtoUtils.fromTJson(message.getPartnerResult(), OrderInfoResponse.class);
            storeRefundStatus(refund, orderInfoResponse);
        }
        starTrekService.createIssueForTrainTicketRefundFailed(refund.getOrderItem().getOrder(), refund, context);
        context.setState(ETrainTicketRefundState.RS_FAILED);
        List<Integer> blankIds = refund.getPayload().getItems().stream().map(PassengerRefundInfo::getBlankId)
                .collect(Collectors.toList());
        context.scheduleExternalEvent(refund.getOrderItemWorkflowId(), TRefundingTicketFailed.newBuilder()
                .setRefundId(refund.getId().toString())
                .addAllBlankIds(blankIds).build());
    }

    private void storeRefundStatus(TrainTicketRefund refund, OrderInfoResponse orderInfoResponse) {
        var referenceIds = refund.getPayload().getItems().stream()
                .filter(x -> x.getRefundReferenceId() != null)
                .map(PassengerRefundInfo::getRefundReferenceId)
                .collect(Collectors.toSet());
        Map<String, OrderItemResponse> refundOperations = orderInfoResponse.getOrderItems().stream()
                .filter(x -> x.getAgentReferenceId() != null && referenceIds.contains(x.getAgentReferenceId()))
                .collect(Collectors.toMap(OrderItemResponse::getAgentReferenceId, x -> x));
        Map<Integer, OrderItemBlank> refundBlanks = orderInfoResponse.getOrderItems().stream()
                .filter(x -> x.getAgentReferenceId() != null && referenceIds.contains(x.getAgentReferenceId()))
                .flatMap(x -> x.getOrderItemBlanks().stream())
                .collect(Collectors.toMap(OrderItemBlank::getPreviousOrderItemBlankId, x -> x));
        for (PassengerRefundInfo i : refund.getPayload().getItems()) {
            OrderItemResponse refundOperation = refundOperations.get(i.getRefundReferenceId());
            OrderItemBlank refundBlank = refundBlanks.get(i.getBlankId());
            if (refundOperation == null) {
                i.setRefundOperationStatus(ImOperationStatus.IN_PROCESS);
            } else {
                i.setRefundOperationStatus(refundOperation.getSimpleOperationStatus());
                i.setRefundOperationId(refundOperation.getOrderItemId());
                if (i.isDependent()) {
                    i.setActualRefundTicketAmount(Money.zero(ProtoCurrencyUnit.RUB));
                    if (refundBlank != null) {
                        i.setRefundBlankId(refundBlank.getOrderItemBlankId());
                        i.setBlankStatus(refundBlank.getBlankStatus());
                    }
                } else if (refundBlank == null) {
                    if (refundOperation.getSimpleOperationStatus() == ImOperationStatus.OK) {
                        throw new RuntimeException(String.format("Refund status is OK, but blank %s not found", i.getBlankId()));
                    }
                } else {
                    i.setRefundBlankId(refundBlank.getOrderItemBlankId());
                    i.setBlankStatus(refundBlank.getBlankStatus());
                    i.setActualRefundTicketAmount(Money.of(refundBlank.getAmount(), ProtoCurrencyUnit.RUB));
                    // TODO(ganintsev): notify developers if !i.getActualRefundTicketAmount().equals(i.getCalculatedRefundTicketAmount())
                }
            }
        }
    }
}
