package ru.yandex.travel.orders.workflows.orderitem.train.ticketrefund.handlers;

import java.time.Instant;
import java.util.List;
import java.util.UUID;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.TrainTicketRefund;
import ru.yandex.travel.orders.services.train.ImClientProvider;
import ru.yandex.travel.orders.workflow.orderitem.train.ticketrefund.proto.ETrainTicketRefundState;
import ru.yandex.travel.orders.workflow.orderitem.train.ticketrefund.proto.TRefundAll;
import ru.yandex.travel.orders.workflow.orderitem.train.ticketrefund.proto.TRefundOne;
import ru.yandex.travel.train.model.refund.PassengerRefundInfo;
import ru.yandex.travel.train.partners.im.ImClientException;
import ru.yandex.travel.train.partners.im.model.AutoReturnRequest;
import ru.yandex.travel.train.partners.im.model.RailwayAutoReturnRequest;
import ru.yandex.travel.train.partners.im.model.orderinfo.ImOperationStatus;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
@RequiredArgsConstructor
public class RefundingStateHandler
        extends AnnotatedStatefulWorkflowEventHandler<ETrainTicketRefundState, TrainTicketRefund> {
    private static final int REFERENCE_ID_MAX_LENGTH = 32;
    private final ImClientProvider imClientProvider;

    @HandleEvent
    public void handleRefundOne(TRefundOne message,
                                StateContext<ETrainTicketRefundState, TrainTicketRefund> context) {
        TrainTicketRefund refund = context.getWorkflowEntity();
        PassengerRefundInfo itemToReturn = getItemToReturn(refund);

        String referenceId = createReferenceId(refund, String.valueOf(itemToReturn.getBlankId()), context.getAttempt());
        var imReq = new AutoReturnRequest();
        imReq.setServiceAutoReturnRequest(new RailwayAutoReturnRequest());
        imReq.getServiceAutoReturnRequest().setOrderItemBlankIds(List.of(itemToReturn.getBlankId()));
        imReq.getServiceAutoReturnRequest().setAgentReferenceId(referenceId);
        Integer buyOperationId = itemToReturn.getBuyOperationId() != null ? itemToReturn.getBuyOperationId() :
                refund.getPayload().getBuyOperationId();
        imReq.getServiceAutoReturnRequest().setOrderItemId(buyOperationId);
        if (Strings.isNullOrEmpty(itemToReturn.getCheckDocumentNumber())) {
            imReq.getServiceAutoReturnRequest().setCheckDocumentNumber(refund.getPayload().getCheckDocumentNumber());
        } else {
            imReq.getServiceAutoReturnRequest().setCheckDocumentNumber(itemToReturn.getCheckDocumentNumber());
        }
        for (PassengerRefundInfo item: refund.getPayload().getItems()) {
            if (item.getBlankId() == itemToReturn.getBlankId()) {
                item.setRefundReferenceId(referenceId);
                item.setRefundOperationStatus(ImOperationStatus.IN_PROCESS);
            }
        }
        try {
            imClientProvider.getImClientForOrderItem(refund.getOrderItem()).autoReturn(imReq);
        } catch (ImClientException e) {
            // never retry autoReturn
            logBlankRefundError(refund.getId(), String.valueOf(itemToReturn.getBlankId()), e);
        }

        if (getItemToReturn(refund) != null) {
            context.scheduleEvent(TRefundOne.newBuilder().build());
        } else {
            refundingComplete(refund, context);
        }
    }

    @HandleEvent
    public void handleRefundAll(TRefundAll message,
                                StateContext<ETrainTicketRefundState, TrainTicketRefund> context) {
        TrainTicketRefund refund = context.getWorkflowEntity();

        var referenceId = createReferenceId(refund, "all", context.getAttempt());
        List<Integer> buyOperationIds = refund.getPayload().getBuyOperationIds();
        Preconditions.checkState(buyOperationIds.size() == 1);
        var imReq = new AutoReturnRequest();
        imReq.setServiceAutoReturnRequest(new RailwayAutoReturnRequest());
        imReq.getServiceAutoReturnRequest().setAgentReferenceId(referenceId);
        imReq.getServiceAutoReturnRequest().setOrderItemId(buyOperationIds.get(0));
        imReq.getServiceAutoReturnRequest().setCheckDocumentNumber(refund.getPayload().getCheckDocumentNumber());
        refund.getPayload().getItems().forEach(b -> {
            b.setRefundReferenceId(referenceId);
            b.setRefundOperationStatus(ImOperationStatus.IN_PROCESS);
        });
        try {
            imClientProvider.getImClientForOrderItem(refund.getOrderItem()).autoReturn(imReq);
        } catch (ImClientException e) {
            // never retry autoReturn
            logBlankRefundError(refund.getId(), "all", e);
        }

        refundingComplete(refund, context);
    }

    private void refundingComplete(TrainTicketRefund refund,
                                   StateContext<ETrainTicketRefundState, TrainTicketRefund> context) {
        refund.setBackgroundJobActive(true);
        refund.setNextCheckAt(Instant.now());
        context.setState(ETrainTicketRefundState.RS_CHECKING_REFUND);
    }

    private void logBlankRefundError(UUID refundId, String blankId, Exception e) {
        log.error("Error occurred while trying to refund blank={}, refundId={}", blankId, refundId, e);
    }

    private static PassengerRefundInfo getItemToReturn(TrainTicketRefund refund) {
        return refund.getPayload().getItems().stream()
                .filter(x -> x.getRefundReferenceId() == null)
                .findFirst().orElse(null);
    }

    private static String createReferenceId(TrainTicketRefund refund, String suffix, Integer tryNumber) {
        String result = refund.getId().toString() + ":" + suffix;
        if (tryNumber > 0) {
            result += ":" + tryNumber;
        }
        if (result.length() > REFERENCE_ID_MAX_LENGTH) {
            result = result.substring(result.length() - REFERENCE_ID_MAX_LENGTH);
        }
        return result;
    }
}
