package ru.yandex.travel.orders.workflows.orderitem.travelline.handlers;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.common.orders.TravellineHotelItinerary;
import ru.yandex.travel.hotels.common.partners.base.CallContext;
import ru.yandex.travel.hotels.common.partners.travelline.TravellineClient;
import ru.yandex.travel.hotels.common.partners.travelline.exceptions.ReturnedErrorException;
import ru.yandex.travel.hotels.common.partners.travelline.model.BookingStatus;
import ru.yandex.travel.hotels.common.partners.travelline.model.ErrorType;
import ru.yandex.travel.orders.entities.TravellineOrderItem;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.workflow.hotels.proto.TCancellationStart;
import ru.yandex.travel.orders.workflow.hotels.proto.TConfirmationStart;
import ru.yandex.travel.orders.workflow.hotels.travelline.proto.ETravellineItemState;
import ru.yandex.travel.orders.workflow.hotels.travelline.proto.TCancellationCommit;
import ru.yandex.travel.orders.workflow.hotels.travelline.proto.TReservationExpired;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;

@Slf4j
@IgnoreEvents(types = {TCancellationStart.class, TConfirmationStart.class, TReservationExpired.class})
public class CancellingStateHandler extends BaseTravellineHandler {
    private final TravellineClient client;

    public CancellingStateHandler(TravellineClient client, Meters meters) {
        super(meters);
        this.client = client;
    }

    @HandleEvent
    public void handleCancellationCommit(TCancellationCommit message,
                                         StateContext<ETravellineItemState, TravellineOrderItem> context) {
        TravellineHotelItinerary itinerary = context.getWorkflowEntity().getItinerary();
        Preconditions.checkNotNull(itinerary.getYandexNumber());
        Preconditions.checkNotNull(itinerary.getTravellineNumber());
        log.info("Looking up existing reservation");
        var readReservationResponse = wrap(() -> client
                .withCallContext(getCallContext(context.getWorkflowEntity(), CallContext.CallPhase.ORDER_CANCELLATION))
                .readReservationSync(itinerary.getYandexNumber()), true);
        Preconditions.checkNotNull(readReservationResponse, "Reservation not found");
        Preconditions.checkArgument(readReservationResponse.getHotelReservations().size() == 1,
                "Unexpected number of hotel reservations");
        var reservation = readReservationResponse.getHotelReservations().get(0);
        log.info("existing reservation found with reservation code #{} and status {}",
                reservation.getReservationNumber(), reservation.getStatus());
        if (reservation.getStatus() == BookingStatus.CANCELLED) {
            log.info("reservation is already cancelled");
        } else {
            log.info("calling Travelline cancellation API");
            try {
                var rsp = wrap(() -> client
                        .withCallContext(getCallContext(context.getWorkflowEntity(),
                                CallContext.CallPhase.ORDER_CANCELLATION))
                        .cancelReservationSync(itinerary.getYandexNumber()), true);
                Preconditions.checkArgument(rsp.getHotelReservations().size() == 1,
                        "Unexpected number of hotel reservations");
                Preconditions.checkArgument(rsp.getHotelReservations().get(0).getStatus() == BookingStatus.CANCELLED,
                        String.format("Unexpected reservation status '%s'",
                                rsp.getHotelReservations().get(0).getStatus()));
            } catch (ReturnedErrorException ex) {
                if (ex.getErrors().size() == 1 && ex.getErrors().get(0).getErrorCode() == ErrorType.BOOKING_ALREADY_CANCELLED) {
                    log.info("reservation is already cancelled");
                } else {
                    throw ex;
                }
            }
        }
        log.info("CANCELLED");
        context.setState(ETravellineItemState.IS_CANCELLED);
        context.scheduleExternalEvent(context.getWorkflowEntity().getOrderWorkflowId(),
                TServiceCancelled.newBuilder().setServiceId(context.getWorkflowEntity().getId().toString()).build());
    }
}
